<?php

namespace Drupal\Tests\tmgmt_smartling\Kernel;

use Smartling\AuditLog\Params\CreateRecordParameters;

/**
 * Tests file upload flow.
 *
 * @group tmgmt_smartling_kernel
 */
class UploadFlowTest extends SmartlingTestBase {

  /**
   * Asserts upload parameters.
   *
   * PHPUnit callback.
   *
   * @param $subject
   * @return bool
   */
  public function assertUploadParameters($subject) {
    $params = $subject->exportToArray();

    return $params['authorize'] == 0 &&
      preg_match('/^{"client":"drupal-tmgmt-connector","version":"(\d+\.x-\d+\.\d+|\d+\.x-\d+\.x-dev|unknown)"}$/', $params['smartling.client_lib_id']) &&
      $params['localeIdsToAuthorize'][0] == 'de' &&
      $params['smartling.translate_paths'] == 'html/body/div/div, html/body/div/span' &&
      $params['smartling.string_format_paths'] == 'html : html/body/div/div, @default : html/body/div/span' &&
      $params['smartling.variants_enabled'] == TRUE &&
      $params['smartling.source_key_paths'] == 'html/body/div/{div.sl-variant}, html/body/div/{span.sl-variant}' &&
      $params['smartling.character_limit_paths'] == 'html/body/div/limit' &&
      $params['smartling.placeholder_format_custom'] == '(@|%|!)[\w-]+' &&
      $params['smartling.include_translatable_attributes'] == 'title, alt' &&
      $params['smartling.exclude_translatable_attributes'] == '' &&
      $params['smartling.force_block_for_tags'] == '';
  }

  /**
   * Asserts file attachment upload parameters.
   *
   * PHPUnit callback.
   *
   * @param $subject
   * @return bool
   */
  public function assertFileAttachmentUploadParameters($subject) {
    $params = $subject->exportToArray();

    return $params['authorize'] == 0 &&
      preg_match('/^{"client":"drupal-tmgmt-connector","version":"(\d+\.x-\d+\.\d+|\d+\.x-\d+\.x-dev|unknown)"}$/', $params['smartling.client_lib_id']) &&
      $params['localeIdsToAuthorize'][0] == 'de';
  }

  /**
   * Upload success flow with batch execution.
   */
  public function testRequestTranslationSuccessFlowExecute() {
    $translate_job = $this->createJobWithItems([
      'batch_uid' => 'uid',
      'batch_execute_on_job' => 1,
    ]);

    $this->translationRequestManagerMock->expects($this->once())
      ->method('upsertTranslationRequest')
      ->with($translate_job)
      ->willReturn(['translationRequestUid' => 'test']);

    $this->translationRequestManagerMock->expects($this->never())
      ->method('commitError');

    $this->apiWrapperMock->expects($this->once())
      ->method('getApi')
      ->with('batch')
      ->willReturn($this->batchApiMock);

    $this->apiWrapperMock->expects($this->once())
      ->method('createAuditLogRecord')
      ->with(
        $translate_job,
        NULL,
        \Drupal::currentUser(),
        CreateRecordParameters::ACTION_TYPE_UPLOAD
      );

    $this->batchApiMock->expects($this->once())
      ->method('uploadBatchFile')
      ->with(
        'public://tmgmt_sources/JobID1_en_de.xml',
        'JobID1_en_de.xml',
        'xml',
        'uid',
        $this->callback(function($subject) {
          return $this->assertUploadParameters($subject);
        })
      );

    $this->apiWrapperMock->expects($this->once())
      ->method('executeBatch')
      ->with('uid');

    $this->translationRequestManagerMock->expects($this->once())
      ->method('commitSuccessfulUpload')
      ->with($translate_job)
      ->willReturn(FALSE);

    $translate_job->requestTranslation();
  }

  /**
   * Upload file attachment success flow.
   */
  public function testRequestTranslationAttahcmentSuccessFlow() {
    $translate_job = $this->createJobWithItems([
      'batch_uid' => 'uid',
      'batch_execute_on_job' => 1,
    ]);

    $file = $this->createFile("test.txt");

    $this->translationRequestManagerMock->expects($this->once())
      ->method('upsertTranslationRequest')
      ->with($translate_job, $file)
      ->willReturn(['translationRequestUid' => 'test']);

    $this->translationRequestManagerMock->expects($this->never())
      ->method('commitError');

    $this->apiWrapperMock->expects($this->once())
      ->method('getApi')
      ->with('batch')
      ->willReturn($this->batchApiMock);

    $this->apiWrapperMock->expects($this->never())
      ->method('createAuditLogRecord');

    $this->batchApiMock->expects($this->once())
      ->method('uploadBatchFile')
      ->with(
        'public://tmgmt_sources/test.txt',
        'JobID1_en_de_test.txt',
        'plain_text',
        'uid',
        $this->callback(function($subject) {
          return $this->assertFileAttachmentUploadParameters($subject);
        })
      );

    $this->apiWrapperMock->expects($this->never())
      ->method('executeBatch');

    $this->translationRequestManagerMock->expects($this->once())
      ->method('commitSuccessfulUpload')
      ->with($translate_job)
      ->willReturn(FALSE);

    $translate_job->getTranslatorPlugin()->requestTranslationAttachment($translate_job, $file);
  }

  /**
   * Upload with attachments success flow with batch execution.
   */
  public function testRequestTranslationWithAttachments() {
    $translate_job = $this->createJobWithItems([
      'batch_uid' => 'uid',
      'batch_execute_on_job' => 1,
    ]);

    $file = $this->createFile("test.txt");

    $this->createAttachmentFileJobItem($translate_job, $file);

    $this->translationRequestManagerMock->expects($this->never())
      ->method('commitError');

    $this->translationRequestManagerMock->expects($this->exactly(2))
      ->method('upsertTranslationRequest')
      ->willReturnCallback(function($job_param, $file_param) use ($translate_job, $file) {
        // 1st call.
        if ($file_param === NULL && $job_param->id() === $translate_job->id()) {
          return ['translationRequestUid' => 'test'];
        }

        // 2nd call.
        if ($file_param !== NULL && $job_param->id() === $translate_job->id() && $file->id() === $file_param->id()) {
          return ['translationRequestUid' => 'test_attachment'];
        }
      });

    $this->apiWrapperMock->expects($this->exactly(2))
      ->method('getApi')
      ->with('batch')
      ->willReturn($this->batchApiMock);

    $this->batchApiMock->expects($this->exactly(2))
      ->method('uploadBatchFile')
      ->willReturnCallback(function($uri, $filename, $extension, $uid, $params) {
        // 1st call.
        if ($uri === "public://tmgmt_sources/JobID1_en_de.xml") {
          $this->assertEquals($uri, 'public://tmgmt_sources/JobID1_en_de.xml');
          $this->assertEquals($filename, 'JobID1_en_de.xml');
          $this->assertEquals($extension, 'xml');
          $this->assertEquals($uid, 'uid');
          $this->assertTrue($this->assertUploadParameters($params));
        }

        // 2st call.
        if ($uri === "public://tmgmt_sources/test.txt") {
          $this->assertEquals($uri, 'public://tmgmt_sources/test.txt');
          $this->assertEquals($filename, 'JobID1_en_de_test.txt');
          $this->assertEquals($extension, 'plain_text');
          $this->assertEquals($uid, 'uid');
          $this->assertTrue($this->assertFileAttachmentUploadParameters($params));
        }
      });

    $this->apiWrapperMock->expects($this->once())
      ->method('executeBatch')
      ->with('uid');

    $translate_job->requestTranslation();
  }

  /**
   * Upload success flow without batch execution.
   */
  public function testRequestTranslationSuccessFlowDoNotExecute() {
    $translate_job = $this->createJobWithItems([
      'batch_uid' => 'uid',
      'batch_execute_on_job' => 2,
    ]);

    $this->translationRequestManagerMock->expects($this->once())
      ->method('upsertTranslationRequest')
      ->with($translate_job)
      ->willReturn(['translationRequestUid' => 'test']);

    $this->translationRequestManagerMock->expects($this->never())
      ->method('commitError');

    $this->apiWrapperMock->expects($this->once())
      ->method('getApi')
      ->with('batch')
      ->willReturn($this->batchApiMock);

    $this->apiWrapperMock->expects($this->once())
      ->method('createAuditLogRecord')
      ->with(
        $translate_job,
        NULL,
        \Drupal::currentUser(),
        CreateRecordParameters::ACTION_TYPE_UPLOAD
      );

    $this->batchApiMock->expects($this->once())
      ->method('uploadBatchFile')
      ->with(
        'public://tmgmt_sources/JobID1_en_de.xml',
        'JobID1_en_de.xml',
        'xml',
        'uid',
        $this->callback(function($subject) {
          return $this->assertUploadParameters($subject);
        })
      );

    $this->apiWrapperMock->expects($this->never())
      ->method('executeBatch')
      ->with('uid');

    $this->translationRequestManagerMock->expects($this->once())
      ->method('commitSuccessfulUpload')
      ->with($translate_job)
      ->willReturn(TRUE);

    $this->apiWrapperMock->expects($this->once())
      ->method('createFirebaseRecord')
      ->with('tmgmt_smartling', 'notifications', 10, [
        "message" => 'File uploaded. Job id: 1, file name: JobID1_en_de.xml.',
        "type" => "status",
      ]);

    $translate_job->requestTranslation();
  }

  /**
   * Upload fail flow: no batch uid.
   */
  public function testRequestTranslationFailFlowNoBatchUid() {
    $this->translationRequestManagerMock->expects($this->never())
      ->method('upsertTranslationRequest');

    $this->translationRequestManagerMock->expects($this->never())
      ->method('commitError');

    $translate_job = $this->createJobWithItems([]);

    $this->apiWrapperMock->expects($this->once())
      ->method('createAuditLogRecord')
      ->with(
        $translate_job,
        NULL,
        \Drupal::currentUser(),
        CreateRecordParameters::ACTION_TYPE_UPLOAD
      );

    $this->apiWrapperMock->expects($this->never())
      ->method('getApi');

    $this->batchApiMock->expects($this->never())
      ->method('uploadBatchFile');

    $this->apiWrapperMock->expects($this->never())
      ->method('executeBatch');

    $this->translationRequestManagerMock->expects($this->never())
      ->method('commitSuccessfulUpload');

    $this->apiWrapperMock->expects($this->once())
      ->method('createFirebaseRecord')
      ->with('tmgmt_smartling', 'notifications', 10, [
        "message" => "File JobID1_en_de.xml (job id = 1) wasn't uploaded. Please see logs for more info.",
        "type" => "error",
      ]);

    $translate_job->requestTranslation();
  }

  /**
   * Upload fail flow: translation request upsert failed.
   */
  public function testRequestTranslationFailFlowTranslationRequestUpsertFailed() {
    $translate_job = $this->createJobWithItems([
      'batch_uid' => 'uid',
      'batch_execute_on_job' => 1,
    ]);

    $this->translationRequestManagerMock->expects($this->once())
      ->method('upsertTranslationRequest')
      ->with($translate_job)
      ->willReturn(FALSE);

    $this->translationRequestManagerMock->expects($this->never())
      ->method('commitError');

    $this->apiWrapperMock->expects($this->never())
      ->method('getApi');

    $this->batchApiMock->expects($this->never())
      ->method('uploadBatchFile');

    $this->apiWrapperMock->expects($this->never())
      ->method('executeBatch');

    $this->translationRequestManagerMock->expects($this->never())
      ->method('commitSuccessfulUpload');

    $this->apiWrapperMock->expects($this->once())
      ->method('createFirebaseRecord')
      ->with('tmgmt_smartling', 'notifications', 10, [
        "message" => "Can't upsert translation request. File JobID1_en_de.xml (job id = 1) wasn't uploaded. Please see logs for more info.",
        "type" => "error",
      ]);

    $translate_job->requestTranslation();
  }

  /**
   * Upload fail flow: error while uploading.
   */
  public function testRequestTranslationFailFlowErrorWhileUploading() {
    $exception = new \Exception("Test");
    $translation_request = ['translationRequestUid' => 'test'];
    $translate_job = $this->createJobWithItems([
      'batch_uid' => 'uid',
      'batch_execute_on_job' => 1,
    ]);

    $this->translationRequestManagerMock->expects($this->once())
      ->method('upsertTranslationRequest')
      ->with($translate_job)
      ->willReturn($translation_request);

    $this->apiWrapperMock->expects($this->once())
      ->method('getApi')
      ->with('batch')
      ->willReturn($this->batchApiMock);

    $this->apiWrapperMock->expects($this->once())
      ->method('createAuditLogRecord')
      ->with(
        $translate_job,
        NULL,
        \Drupal::currentUser(),
        CreateRecordParameters::ACTION_TYPE_UPLOAD
      );

    $this->apiWrapperMock->expects($this->once())
      ->method('executeBatch')
      ->with('uid')
      ->will($this->throwException($exception));

    $this->translationRequestManagerMock->expects($this->never())
      ->method('commitSuccessfulUpload');

    $this->translationRequestManagerMock->expects($this->once())
      ->method('commitError')
      ->with($translate_job, $translation_request, $exception);

    $translate_job->requestTranslation();
  }

}
