<?php

namespace Drupal\tmgmt_supertext_ai\Plugin\tmgmt\Translator;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\tmgmt\Data;
use Drupal\tmgmt\JobItemInterface;
use Drupal\tmgmt\TMGMTException;
use Drupal\tmgmt\TranslatorPluginBase;
use Drupal\tmgmt_supertext_ai\SupertextAiData;
use GuzzleHttp\Client;
use Psr\Http\Message\ResponseInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\tmgmt\TranslatorInterface;
use Drupal\tmgmt\JobInterface;
use Drupal\tmgmt\Translator\AvailableResult;

/**
 * Supertext AI translation plugin controller.
 *
 * @TranslatorPlugin(
 *   id = "supertext_ai",
 *   label = @Translation("Supertext AI translator"),
 *   description = @Translation("Supertext AI translation service."),
 *   ui = "Drupal\tmgmt_supertext_ai\SupertextAITranslatorUi",
 *   logo = "icons/supertext.png",
 * )
 */
class SupertextAITranslator extends TranslatorPluginBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a SupertextAITranslator object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param array $plugin_definition
   *   The plugin implementation definition.
   * @param \Psr\Log\LoggerInterface $logger
   *   Logger for this channel.
   * @param \GuzzleHttp\Client $httpClient
   *   Http client.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   Current user.
   * @param \Drupal\tmgmt\Data $dataHelper
   *   The data service.
   */
  public function __construct(
    $configuration,
    $plugin_id,
    array $plugin_definition,
    protected LoggerInterface $logger,
    protected Client $httpClient,
    protected AccountProxyInterface $currentUser,
    protected Data $dataHelper) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('logger.factory')->get('tmgmnt_supertext'),
      $container->get('http_client'),
      $container->get('current_user'),
      $container->get('tmgmt.data')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function checkAvailable(TranslatorInterface $translator): AvailableResult {
    if ($translator->getSetting('api_key')) {
      return AvailableResult::yes();
    }
    return AvailableResult::no(t('@translator is not available. Make sure it is properly <a href=:configured>configured</a>.', [
      '@translator' => $translator->label(),
      ':configured' => $translator->toUrl()->toString(),
    ]));
  }

  /**
   * {@inheritdoc}
   */
  public function requestTranslation(JobInterface $job): void {
    foreach ($job->getItems() as $job_item) {
      $this->requestJobItemTranslation($job_item);
    }
    if (!$job->isRejected()) {
      $job->submitted('The translation job has been submitted.');
    }
  }

  /**
   * Request job item translations.
   *
   * @param JobItemInterface $job_item
   *   The job item.
   *
   * @return void
   */
  public function requestJobItemTranslation(JobItemInterface $job_item): void {
    $job = $job_item->getJob();

    $chunks = new SupertextAiData($job->getSourceLangcode(), $job->getRemoteTargetLanguage());
    foreach ($this->dataHelper->filterTranslatable($job_item->getData()) as $path => $text) {
      $chunks->addText($path, $text['#text']);
    }

    try {
      // Loop over the chunked data and request translations for it.
      $translations = [];
      foreach ($chunks as $chunk_id => $chunk) {
        $response = $this->supertextHttpRequest('text', $job->getTranslator(), $chunk);
        $response_data = json_decode($response->getBody()->getContents(), TRUE, 512, JSON_THROW_ON_ERROR);

        $chunk_mapping = $chunks->getMapping($chunk_id);
        foreach ($response_data['translated_text'] as $key => $text) {
          if (!isset($translations[$chunk_mapping[$key]]['#text'])) {
            $translations[$chunk_mapping[$key]]['#text'] = '';
          }
          $translations[$chunk_mapping[$key]]['#text'] .= $text;
        }
      }

      $job_item->addTranslatedData($this->dataHelper->unflatten($translations));
    }
    catch (\Exception $e) {
      $job->rejected('Job has been rejected with following error: @error', ['@error' => $e->getMessage()], 'error');
    }
  }

  /**
   * General function for making an HTTP Request to Supertext AI API.
   *
   * @param string $endpoint
   *   The endpoint to be called.
   * @param \Drupal\tmgmt\TranslatorInterface $translator
   *   Object of the translator which should be used.
   * @param array $payload
   *   (optoinal) The payload.
   * @param string $method
   *   (optional) With HTTP Method should be used. Default to POST.
   */
  public function supertextHttpRequest(string $endpoint, TranslatorInterface $translator, array $payload = [], string $method = 'POST'): ResponseInterface {
    $api_key = $translator->getSetting('api_key');
    if (!$api_key) {
      throw new TMGMTException(t('No api key found.'));
    }

    $options['headers'] = [
      'User-Agent' => 'Drupal Supertext AI TMGMT module 1.x',
      'Authorization' => 'Supertext-Auth-Key ' . $api_key,
      'Content-Type' => 'application/json',
    ];

    $options['body'] = json_encode($payload, JSON_UNESCAPED_SLASHES);

    $api_server = $translator->getSetting('api_server');
    $api_server_other = $translator->getSetting('api_server_other');

    $this->logger->debug('SupertextAI request: @endpoint<br\> @payload', [
      '@payload' => $options['body'],
      '@endpoint' => self::getApiUrl($api_server, $api_server_other) . $endpoint,
    ]);

    return $this->httpClient->request($method, self::getApiUrl($api_server, $api_server_other) . $endpoint, $options);
  }

  /**
   * Get api url.
   *
   * @param string $api_server
   *   Either 'live', 'development', 'tesing' or 'other'.
   * @param string $api_server_other
   *   If 'other' is selected, the url of the server.
   *
   * @return string
   *   The api server url.
   * @throws \Drupal\tmgmt\TMGMTException
   */
  public static function getApiUrl(string $api_server, string $api_server_other): string {
    switch ($api_server) {
      case 'other':
        if (!trim($api_server_other)) {
          throw new TMGMTException('API server URL empty.');
        }
        return rtrim($api_server_other, '/') . '/';

      case 'live':
        return 'https://api.supertext.com/v1/translate/ai/';

      default:
        return "https://api.$api_server.supertext.com/v1/translate/ai/";
    }


  }

}
