<?php

namespace Drupal\tmgmt_tolgee\Plugin\tmgmt\Translator;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\tmgmt\JobInterface;
use Drupal\tmgmt\TranslatorPluginBase;
use Drupal\tmgmt\TranslatorInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use GuzzleHttp\ClientInterface;
use Drupal\key\KeyRepositoryInterface;
use Drupal\tmgmt\Translator\AvailableResult;
use Drupal\tmgmt\Translator\TranslatableResult;
use Psr\Log\LoggerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Component\Plugin\PluginManagerInterface;
use Drupal\tmgmt\TMGMTException;
use Drupal\Core\State\StateInterface;

/**
 * Tolgee implementation of the TranslatorPluginBase.
 *
 * @TranslatorPlugin(
 *   id = "tolgee",
 *   label = @Translation("Tolgee Translator"),
 *   description = @Translation("Integrates with Tolgee API for shared translations."),
 *   ui = "Drupal\tmgmt_tolgee\TolgeeTranslatorUi",
 *   default_settings = {
 *     "base_url" = "https://app.tolgee.io",
 *     "api_key" = "",
 *     "project_id" = "",
 *     "namespace" = "drupal",
 *     "auto_accept" = FALSE,
 *     "auto_sync" = FALSE,
 *     "import_options" = {
 *       "tag_job" = TRUE,
 *       "tag_source" = TRUE,
 *     }
 *   },
 *   logo = "icons/tolgee.svg",
 * )
 */
class TolgeeTranslator extends TranslatorPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $client;

  /**
   * The Key Repository.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected $keyRepository;

  /**
   * The File System service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The TMGMT File Format Manager.
   *
   * @var \Drupal\Component\Plugin\PluginManagerInterface
   */
  protected $formatManager;

  /**
   * The State service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * Constructs a TolgeeTranslator object.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param \GuzzleHttp\ClientInterface $client
   *   The HTTP client.
   * @param \Drupal\key\KeyRepositoryInterface $key_repository
   *   The Key repository.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The File System service.
   * @param \Drupal\Component\Plugin\PluginManagerInterface $format_manager
   *   The TMGMT Format Manager.
   */
  public function __construct(
      array $configuration,
      $plugin_id,
      $plugin_definition,
      ClientInterface $client,
      KeyRepositoryInterface $key_repository,
      FileSystemInterface $file_system,
      PluginManagerInterface $format_manager,
      StateInterface $state
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->client = $client;
    $this->keyRepository = $key_repository;
    $this->fileSystem = $file_system;
    $this->formatManager = $format_manager;
    $this->state = $state;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('http_client'),
      $container->get('key.repository'),
      $container->get('file_system'),
      $container->get('plugin.manager.tmgmt_file.format'),
      $container->get('state')
    );
  }



  /**
   * Helper to build the base URL for API requests.
   *
   * @param \Drupal\tmgmt\TranslatorInterface $translator
   *   The translator entity.
   *
   * @return string
   *   The base API URL (e.g., https://app.tolgee.io/v2/projects/{id} or https://app.tolgee.io/v2/projects).
   */
  private function getProjectUrl(TranslatorInterface $translator) {
      $base_url = rtrim($translator->getSetting('base_url'), '/');
      $project_id = $translator->getSetting('project_id');
      
      if (!empty($project_id)) {
          return $base_url . '/v2/projects/' . $project_id;
      }
      return $base_url . '/v2/projects';
  }

  /**
   * {@inheritdoc}
   */
  public function checkAvailable(TranslatorInterface $translator) {
    $api_key = $translator->getSetting('api_key');
    if (!$api_key) {
        return AvailableResult::no(t('@translator is not available. Make sure it is properly configured.', ['@translator' => $translator->label()]));
    }
    
    // PAT Validation
    // We need to resolve the key to check its value prefix.
    $key_entity = $this->keyRepository->getKey($api_key);
    if ($key_entity) {
        $key_value = $key_entity->getKeyValue();
        if (str_starts_with($key_value, 'tgpat_') && empty($translator->getSetting('project_id'))) {
             return AvailableResult::no(t('A Project ID is required when using a Personal Access Token (tgpat_).'));
        }
    }
    
    return AvailableResult::yes();
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedRemoteLanguages(TranslatorInterface $translator) {
      $languages = [];
      $key_id = $translator->getSetting('api_key');
      if (!$key_id) {
          return $languages;
      }
      $key_entity = $this->keyRepository->getKey($key_id);
      if (!$key_entity) {
          return $languages;
      }
      $api_key = $key_entity->getKeyValue();
      
      try {
          // Use project URL
          $endpoint = $this->getProjectUrl($translator) . '/languages';
          
          $response = $this->client->request('GET', $endpoint, [
              'headers' => [
                  'X-API-Key' => $api_key,
                  'Accept' => 'application/json',
              ],
              'query' => [
                  'size' => 1000,
              ]
          ]);
          
          $data = json_decode($response->getBody()->getContents(), TRUE);
          
          if (isset($data['_embedded']['languages'])) {
              foreach ($data['_embedded']['languages'] as $lang) {
                  $languages[$lang['tag']] = $lang['tag'];
              }
          }
      } catch (\Exception $e) {
          \Drupal::logger('tmgmt_tolgee')->error('Failed to fetch languages: ' . $e->getMessage());
      }
      
      return $languages;
  }

  /**
   * {@inheritdoc}
   */
  public function checkTranslatable(TranslatorInterface $translator, JobInterface $job) {
    return TranslatableResult::yes();
  }

  /**
   * {@inheritdoc}
   */
/**
   * {@inheritdoc}
   */
  public function requestTranslation(JobInterface $job) {
    /** @var \Drupal\tmgmt_file\Format\FormatManager $format_manager */
    $converter = $this->formatManager->createInstance('xlf');
    
    // Export job data to string
    $content = $converter->export($job);

    // Get settings
    $custom_tags = $job->getSetting('tags');
    $tag_job = $job->getTranslator()->getSetting('import_options')['tag_job'] ?? TRUE;
    $tag_source = $job->getTranslator()->getSetting('import_options')['tag_source'] ?? TRUE;
    
    // Namespace logic
    $namespace = $job->getSetting('namespace');
    
    // Build Tags Array (Consolidated)
    $tags = [];
    if ($tag_job) {
        $tags[] = 'tmgmt-job-' . $job->id();
    }
    if ($tag_source) {
        // Use the source language ID (e.g., 'source-en')
        $tags[] = 'source-' . $job->getSourceLanguage()->getId();
    }
    if (!empty($custom_tags)) {
        // Merge custom tags entered by the user
        $tags = array_merge($tags, array_map('trim', explode(',', $custom_tags)));
    }
    
    // Validate Configuration
    $key_id = $job->getTranslator()->getSetting('api_key');
    $key_entity = $this->keyRepository->getKey($key_id);
    if (!$key_entity) {
         throw new TMGMTException('Tolgee API Key not configured or missing.');
    }
    $api_key = $key_entity->getKeyValue();
    
    // ---------------------------------------------------------
    // FIX: Single-Step Import with File Mappings for Namespace
    // ---------------------------------------------------------
    $endpoint = $this->getProjectUrl($job->getTranslator()) . '/single-step-import';
    $filename = 'job_' . $job->id() . '.xlf';
    $force_mode = $job->getTranslator()->getSetting('import_options')['force_mode'] ?? 'OVERRIDE';
    
    // Build JSON Params object matching the API Spec
    $params = [
        'format' => 'XLIFF',
        'forceMode' => $force_mode,
        'overrideKeyDescriptions' => TRUE,
        'convertPlaceholdersToIcu' => FALSE,
        'createNewKeys' => TRUE,
    ];

    // Namespace is set via fileMappings
    if (!empty($namespace)) {
        $params['fileMappings'] = [
            [
                'fileName' => $filename,
                'namespace' => $namespace,
            ]
        ];
    } else {
        // Even if empty, API says fileMappings is required? 
        // Let's provide an empty mapping or rely on global.
        // If required, we map to empty namespace or skip?
        // User said `fileMappings object[] required`.
        $params['fileMappings'] = [
             [
                'fileName' => $filename,
                // 'namespace' => '', // Optional if global
            ]
        ];
    }

    if (!empty($tags)) {
        $params['tagNewKeys'] = array_values($tags); 
        // tagExistingKeys is not in the list user provided, only tagNewKeys?
        // User pasted: "tagNewKeys string[] required".
        // Let's stick to tagNewKeys to be safe, or check if existing supported.
        // Assuming common sense, we might want both, but let's follow the snippet.
        // "Keys created by this import will be tagged... The keys that already exist will not be tagged."
        // That implies tagExistingKeys is not supported or not listed.
        // Let's just use tagNewKeys for now to avoid errors, or try both if critical.
        // Given earlier attempts, let's keep it simple.
    }
    
    $multipart = [
      [
        'name'     => 'files', 
        'contents' => $content,
        'filename' => $filename,
      ],
      [
        'name' => 'params',
        'contents' => json_encode($params),
        'headers' => [
            'Content-Type' => 'application/json',
        ],
      ],
    ];
    
    try {
      // Single Atomic Request
      $this->client->request('POST', $endpoint, [
        'headers' => [
          'X-API-Key' => $api_key,
          'Accept' => 'application/json',
        ],
        'multipart' => $multipart,
      ]);

      $job->submitted('Job submitted to Tolgee (Single-Step Import).');
      
    } catch (\Exception $e) {
      $job->rejected('Tolgee API Error: ' . $e->getMessage());
      throw new TMGMTException('Tolgee API Error: ' . $e->getMessage(), [], $e->getCode(), $e);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function fetchTranslation(JobInterface $job) {
    /** @var \Drupal\tmgmt_file\Format\FormatManager $format_manager */
    $converter = $this->formatManager->createInstance('xlf');

    $key_id = $job->getTranslator()->getSetting('api_key');
    $key_entity = $this->keyRepository->getKey($key_id);
    if (!$key_entity) {
         throw new TMGMTException('Tolgee API Key not configured or missing.');
    }
    $api_key = $key_entity->getKeyValue();
    
    $endpoint = $this->getProjectUrl($job->getTranslator()) . '/export';
    $target_lang = $job->getTargetLangcode();
    
    // ETag Handling
    $state_key = 'tmgmt_tolgee.etag.' . $job->id();
    $etag = $this->state->get($state_key);
    
    try {
      $options = [
        'headers' => [
          'X-API-Key' => $api_key,
          'Accept' => 'application/json, application/zip',
        ],
        'query' => [
            'format' => 'JSON',
            'languages' => $target_lang,
            'zip' => 'true', 
            'structureDelimiter' => '', 
        ],
      ];
      
      // Add If-None-Match if ETag exists
      if ($etag) {
          $options['headers']['If-None-Match'] = $etag;
      }

      $response = $this->client->request('GET', $endpoint, $options);
      
      // Handle 304 Not Modified
      if ($response->getStatusCode() === 304) {
          $job->addMessage('Tolgee: No changes detected (304 Not Modified).');
          return;
      }
      
      // Save new ETag
      if ($response->hasHeader('ETag')) {
          $new_etag = $response->getHeaderLine('ETag');
          $this->state->set($state_key, $new_etag);
      }

      $content = $response->getBody()->getContents();
      if (empty($content)) {
          throw new TMGMTException('Empty response from Tolgee.');
      }
      
      $tmp_dir = $this->fileSystem->getTempDirectory();
      $tmp_zip = $tmp_dir . '/tolgee_job_' . $job->id() . '.zip';
      file_put_contents($tmp_zip, $content);
          
      $zip = new \ZipArchive();
      $json_content = '';
          
      if ($zip->open($tmp_zip) === TRUE) {
          for ($i = 0; $i < $zip->numFiles; $i++) {
              $filename = $zip->getNameIndex($i);
              if (str_ends_with($filename, '.json')) {
                  $json_content = $zip->getFromIndex($i);
                  break;
              }
          }
          $zip->close();
          
          if (!$json_content) {
               throw new TMGMTException('No JSON file found in Tolgee export ZIP.');
          }
      } else {
          throw new TMGMTException('Failed to open ZIP file.');
      }
      unlink($tmp_zip);
      
      // Parse JSON
      $data_array = json_decode($json_content, TRUE);
      if (json_last_error() !== JSON_ERROR_NONE) {
           throw new TMGMTException('Failed to parse JSON content: ' . json_last_error_msg());
      }

      // Map JSON Keys directly to Drupal Data
      $imported_data = [];
      foreach ($data_array as $key => $value) {
          if (!empty($key) && is_string($value)) {
              $imported_data[$key]['#text'] = $value;
          }
      }
      
      if (!empty($imported_data)) {
           $data = \Drupal::service('tmgmt.data')->unflatten($imported_data);
           $job->addTranslatedData($data);
           $job->addMessage('Successfully imported translation from Tolgee.');
           
           // Auto-Accept Logic
           // Using Core TMGMT 'Auto accept finished translations' setting.
           if ($job->getTranslator()->isAutoAccept()) {
               foreach ($job->getItems() as $item) {
                   if ($item->isNeedsReview()) {
                       $item->acceptTranslation();
                   }
               }
               $job->addMessage('Automatically accepted translations based on settings.');
           }
           
      } else {
           $job->addMessage('No translation data found in the export.', 'warning');
      }

    } catch (\GuzzleHttp\Exception\ClientException $e) {
        // Guzzle throws exception on 304 sometimes if not configured to handle it? 
        // No, typically 304 is not an error in recent Guzzle versions if strictly handled, but standard behavior treats 4xx/5xx as errors.
        // If 304 causes exception, handle it here.
        if ($e->getResponse() && $e->getResponse()->getStatusCode() === 304) {
             $job->addMessage('Tolgee: No changes detected (304 Not Modified) [Exception].');
             return;
        }
        throw new TMGMTException('Tolgee Fetch Error: ' . $e->getMessage(), [], $e->getCode(), $e);
    } catch (\Exception $e) {
      throw new TMGMTException('Tolgee Fetch Error: ' . $e->getMessage(), [], $e->getCode(), $e);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedTargetLanguages(TranslatorInterface $translator, $source_language) {
    // Ideally fetch from API, but for now we accept what the system defines.
    // Or return empty array to allow all.
    return [];
  }

  /**
   * Determines the best namespace for a job based on its items.
   *
   * @param \Drupal\tmgmt\JobInterface $job
   *   The translation job.
   *
   * @return string|null
   *   The detected namespace or NULL.
   */
  public function determineNamespace(JobInterface $job) {
    $items = $job->getItems();
    // We assume the first item is representative of the whole job.
    if ($item = reset($items)) {
      $type = strtolower($item->getItemType());
      $plugin = strtolower($item->getPlugin());
      $label = strtolower($item->label());
      $id = strtolower($item->getItemId());

      // 1. STRUCTURED MAPPING (Configured via Grouped Table UI)
      $mapping_config = $job->getTranslator()->getSetting('namespace_mapping');
      if (is_array($mapping_config)) {
          foreach ($mapping_config as $group) {
              // Ensure group is valid and has namespace + types
              if (!empty($group['namespace']) && !empty($group['types']) && is_array($group['types'])) {
                  // Check if current item type is in this group's types list
                  // We check match against BOTH Item Type (e.g. node) and Plugin (e.g. content) for flexibility
                  // But UI selects Entity Types (so mostly Item Type)
                  if (in_array($type, $group['types'])) {
                      return $group['namespace'];
                  }
              }
          }
      }
      
      // 2. BUILT-IN DEFAULT MAPPING (Fallback)
      // Navigation (Menus)
      if (strpos($label, 'menu') !== FALSE || strpos($type, 'menu') !== FALSE || strpos($id, 'menu') !== FALSE) {
        return 'navigation';
      }

      // Auth (User, Login, Profile)
      if ($type === 'user' || strpos($type, 'user') !== FALSE || 
          strpos($label, 'login') !== FALSE || strpos($label, 'register') !== FALSE || 
          strpos($label, 'password') !== FALSE || strpos($label, 'account') !== FALSE) {
        return 'auth';
      }

      // Forms (Webforms, Contact, Field Config)
      if ($type === 'webform' || $type === 'contact_message' || 
          strpos($type, 'field_config') !== FALSE || strpos($label, 'form') !== FALSE) {
        return 'forms';
      }

      // Content (Nodes, Taxonomy, Media, Blocks)
      if ($plugin === 'content' || $type === 'node' || $type === 'taxonomy_term' || $type === 'taxonomy_vocabulary' || $type === 'media' || strpos($type, 'block') !== FALSE) {
        return 'content';
      }

      // Actions (Generic Verbs)
      if (str_word_count($label) === 1 && in_array($label, ['save', 'edit', 'delete', 'cancel', 'submit', 'back'])) {
        return 'actions';
      }
      
      return NULL;
    }
    return NULL;
  }
}
