<?php

namespace Drupal\tmgmt_tolgee;

use Drupal\Core\Form\FormStateInterface;
use Drupal\tmgmt\TranslatorPluginUiBase;
use Drupal\tmgmt\JobInterface;
use Drupal\tmgmt_tolgee\Plugin\tmgmt\Translator\TolgeeTranslator;
use Drupal\Component\Utility\NestedArray;

/**
 * Tolgee translator UI.
 */
class TolgeeTranslatorUi extends TranslatorPluginUiBase {

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $translator = $form_state->getFormObject()->getEntity();
    
    // --- 1. SETUP WHITELIST (Tagify) ---
    $entity_definitions = \Drupal::entityTypeManager()->getDefinitions();
    $whitelist = [];
    foreach ($entity_definitions as $id => $definition) {
        $group = $definition->getGroup();
        if ($group != 'content' && $group != 'configuration') continue;
        $whitelist[] = $id; 
    }
    sort($whitelist);

    $form['#attached']['library'][] = 'tmgmt_tolgee/admin.settings';
    $form['#attached']['drupalSettings']['tolgee']['whitelist'] = $whitelist;

    // --- 2. STANDARD SETTINGS ---
    $form['base_url'] = [
      '#type' => 'textfield',
      '#title' => t('Tolgee Base URL'),
      '#default_value' => $translator->getSetting('base_url'),
      '#required' => TRUE,
    ];
    $key_repository = \Drupal::service('key.repository');
    $keys = $key_repository->getKeys();
    $options = [];
    foreach ($keys as $id => $key) {
        $options[$id] = $key->label();
    }
    $form['api_key'] = [
      '#type' => 'select',
      '#title' => t('Tolgee API Key'),
      '#options' => $options,
      '#default_value' => $translator->getSetting('api_key'),
      '#required' => TRUE,
      '#empty_option' => t('- Select Key -'),
    ];
    $form['project_id'] = [
      '#type' => 'textfield',
      '#title' => t('Project ID'),
      '#default_value' => $translator->getSetting('project_id'),
    ];
    $form['auto_sync'] = [
      '#type' => 'checkbox',
      '#title' => t('Auto Sync (Cron)'),
      '#default_value' => $translator->getSetting('auto_sync'),
    ];
    $form['enable_namespaces'] = [
      '#type' => 'checkbox',
      '#title' => t('Enable Namespace Management'),
      '#default_value' => $translator->getSetting('enable_namespaces') ?? TRUE,
    ];

    // --- 3. MAPPING TABLE UI ---
    
    $form['mapping_wrapper'] = [
        '#type' => 'container',
        '#attributes' => ['id' => 'tolgee-mapping-wrapper'],
        '#states' => [
            'visible' => [
                ':input[name="settings[enable_namespaces]"]' => ['checked' => TRUE],
            ],
        ],
    ];

    $form['mapping_wrapper']['namespace_mapping'] = [
        '#type' => 'table',
        '#header' => [
            t('Tolgee Namespace'), 
            t('Source Entity Types'), 
            t('Operations'), // New Column for "X"
        ],
        '#empty' => t('No mappings defined. Add one below.'),
        '#attributes' => ['class' => ['tolgee-mapping-table']], 
    ];

    // --- 4. ROW COUNT LOGIC ---
    
    // Initialize count if missing
    if (!$form_state->has('tolgee_rows')) {
        $stored = $translator->getSetting('namespace_mapping') ?: [];
        $count = empty($stored) ? 1 : count($stored);
        $form_state->set('tolgee_rows', $count);
    }
    
    // Handle "Add Another" Click
    $trigger = $form_state->getTriggeringElement();
    if ($trigger && end($trigger['#parents']) === 'add_item') {
        $count = $form_state->get('tolgee_rows') + 1;
        $form_state->set('tolgee_rows', $count);
    }
    
    // Ensure we never have < 1 row unless intended, 
    // but usually "Add Another" forms keep 1 empty row.
    $num_rows = $form_state->get('tolgee_rows');
    if ($num_rows < 1) {
        $num_rows = 1;
        $form_state->set('tolgee_rows', 1);
    }
    
    // --- 5. DATA RETRIEVAL ---
    
    $db_data = $translator->getSetting('namespace_mapping') ?: [];
    $db_data = array_values($db_data); 
    
    $input = $form_state->getUserInput();
    $user_data = $input['settings']['mapping_wrapper']['namespace_mapping'] 
              ?? $input['mapping_wrapper']['namespace_mapping'] 
              ?? [];

    // --- 6. BUILD ROWS ---
    for ($i = 0; $i < $num_rows; $i++) {
        $ns_val = $user_data[$i]['namespace'] ?? $db_data[$i]['namespace'] ?? '';
        
        $types_raw = $user_data[$i]['types'] ?? $db_data[$i]['types'] ?? '';
        $types_val = is_array($types_raw) ? implode(',', $types_raw) : $types_raw;

        // Column 1: Namespace
        $form['mapping_wrapper']['namespace_mapping'][$i]['namespace'] = [
            '#type' => 'textfield',
            '#default_value' => $ns_val,
            '#size' => 20,
            '#attributes' => ['placeholder' => 'e.g. navigation'],
        ];

        // Column 2: Types (Tagify)
        $form['mapping_wrapper']['namespace_mapping'][$i]['types'] = [
            '#type' => 'textfield', 
            '#default_value' => $types_val,
            '#maxlength' => 2048, 
            '#attributes' => [
                'class' => ['js-tolgee-source-types'], 
                'placeholder' => t('Select types...'),
                'style' => 'display: none;', // Server-side hiding
            ],
        ];

        // Column 3: Remove Button
        $form['mapping_wrapper']['namespace_mapping'][$i]['remove'] = [
            '#type' => 'submit',
            '#value' => '×', // The "X"
            '#name' => 'tolgee_remove_' . $i,
            '#submit' => [[$this, 'removeOne']],
            '#limit_validation_errors' => [],
            '#ajax' => [
                'callback' => [$this, 'addmoreCallback'],
                'wrapper' => 'tolgee-mapping-wrapper',
            ],
            '#attributes' => [
                'class' => ['button', 'button--small'],
                'title' => t('Remove this row'),
                'style' => 'border-radius: 50%; width: 30px; height: 30px; padding: 0; line-height: 28px; text-align: center;',
            ],
        ];
    }

    $form['mapping_wrapper']['add_item'] = [
        '#type' => 'submit',
        '#value' => t('Add another mapping'),
        '#submit' => [[$this, 'addOne']],
        '#limit_validation_errors' => [], 
        '#ajax' => [
            'callback' => [$this, 'addmoreCallback'],
            'wrapper' => 'tolgee-mapping-wrapper',
        ],
    ];
    
    // --- 7. REMAINING FIELDS ---
    $form['namespace'] = [
      '#type' => 'textfield',
      '#title' => t('Default Namespace'),
      '#default_value' => $translator->getSetting('namespace'),
      '#states' => [
        'visible' => [
          ':input[name="settings[enable_namespaces]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['import_options'] = [
        '#type' => 'details',
        '#title' => t('Import Options'),
        '#open' => TRUE,
    ];
    $settings = $translator->getSettings();
    $form['import_options']['tag_job'] = [
        '#type' => 'checkbox',
        '#title' => t('Tag with TMGMT Job ID'),
        '#default_value' => $settings['import_options']['tag_job'] ?? TRUE,
    ];
    $form['import_options']['tag_source'] = [
        '#type' => 'checkbox',
        '#title' => t('Tag with Source Plugin ID'),
        '#default_value' => $settings['import_options']['tag_source'] ?? TRUE,
    ];
    $form['import_options']['force_mode'] = [
        '#type' => 'radios',
        '#title' => t('Conflict Resolution (Force Mode)'),
        '#options' => [
            'OVERRIDE' => t('Override - Update existing keys.'),
            'KEEP' => t('Keep - Do not overwrite.'),
        ],
        '#default_value' => $settings['import_options']['force_mode'] ?? 'OVERRIDE',
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
      parent::validateConfigurationForm($form, $form_state);

      $values = $form_state->getValue('settings');
      
      if (isset($values['mapping_wrapper']['namespace_mapping'])) {
          $mappings = $values['mapping_wrapper']['namespace_mapping'];
          $clean_mappings = []; 

          foreach ($mappings as $row) {
              // Convert CSV string to Array
              if (isset($row['types']) && is_string($row['types'])) {
                  $row['types'] = array_values(array_filter(explode(',', $row['types'])));
              }
              // Skip the 'remove' button value and check content
              if (!empty($row['namespace']) || !empty($row['types'])) {
                  // Ensure we don't save the button value
                  unset($row['remove']); 
                  $clean_mappings[] = $row;
              }
          }
          
          $values['namespace_mapping'] = $clean_mappings;
          unset($values['mapping_wrapper']); 
          $form_state->setValue('settings', $values);
      }
  }

  /**
   * Submit Handler: Add Row
   */
  public function addOne(array &$form, FormStateInterface $form_state) {
    $form_state->setRebuild();
  }

  /**
   * Submit Handler: Remove Row
   */
  public function removeOne(array &$form, FormStateInterface $form_state) {
    // 1. Find which button was clicked
    $trigger = $form_state->getTriggeringElement();
    $parents = $trigger['#parents'];
    
    // Structure is [settings, mapping_wrapper, namespace_mapping, INDEX, remove]
    // We look for 'namespace_mapping' and take the next key
    $key_index = array_search('namespace_mapping', $parents);
    
    if ($key_index !== FALSE && isset($parents[$key_index + 1])) {
        $index_to_remove = $parents[$key_index + 1];

        // 2. Manipulate User Input (Must update input so rebuild sees the change)
        $user_input = $form_state->getUserInput();
        
        // Find the input array location (tmgmt allows variable nesting)
        $path = [];
        if (isset($user_input['settings']['mapping_wrapper']['namespace_mapping'])) {
            $path = ['settings', 'mapping_wrapper', 'namespace_mapping'];
        } elseif (isset($user_input['mapping_wrapper']['namespace_mapping'])) {
            $path = ['mapping_wrapper', 'namespace_mapping'];
        }

        if (!empty($path)) {
            // Get current data
            $data = NestedArray::getValue($user_input, $path);
            
            // Remove the specific row
            if (isset($data[$index_to_remove])) {
                unset($data[$index_to_remove]);
            }
            
            // Re-index array so it is 0, 1, 2 again
            $data = array_values($data);
            
            // Update input and state
            NestedArray::setValue($user_input, $path, $data);
            $form_state->setUserInput($user_input);
            $form_state->set('tolgee_rows', count($data));
        }
    }

    $form_state->setRebuild();
  }

  /**
   * AJAX Callback
   */
  public function addmoreCallback(array &$form, FormStateInterface $form_state) {
    // Dynamic Parent Discovery (Finds the wrapper automatically)
    $trigger = $form_state->getTriggeringElement();
    // Go up 2 levels (remove_button -> row -> table -> wrapper) or look for wrapper ID
    // Safer: Look for the specific wrapper element in the form
    if (isset($form['settings']['mapping_wrapper'])) {
        return $form['settings']['mapping_wrapper'];
    }
    if (isset($form['mapping_wrapper'])) {
        return $form['mapping_wrapper'];
    }
    // Fallback if triggered deeper
    $parents = $trigger['#array_parents'];
    // search for mapping_wrapper key
    $wrapper_index = array_search('mapping_wrapper', $parents);
    if ($wrapper_index !== FALSE) {
        $slice = array_slice($parents, 0, $wrapper_index + 1);
        return NestedArray::getValue($form, $slice);
    }
    return NULL;
  }
  
  public function checkoutSettingsForm(array $form, FormStateInterface $form_state, JobInterface $job) {
      // (Standard implementation)
      $form['tags'] = [
          '#type' => 'textfield',
          '#title' => t('Additional Tags (Labels)'),
          '#description' => t('Optional: Add comma-separated tags to this export (e.g., \"urgent, marketing\").'),
      ];
      $translator_plugin = $job->getTranslator()->getPlugin();
      $namespace = $job->getSetting('namespace');
      $default_namespace = $job->getTranslator()->getSetting('namespace');
      if ((empty($namespace) || $namespace === $default_namespace) && $translator_plugin instanceof TolgeeTranslator) {
          $enable_namespaces = $job->getTranslator()->getSetting('enable_namespaces') ?? TRUE;
          if ($enable_namespaces) {
              $detected = $translator_plugin->determineNamespace($job);
              if ($detected) {
                  $namespace = $detected;
              } elseif (empty($namespace)) {
                  $namespace = $default_namespace;
              }
          }
      }
      $form['namespace'] = [
          '#type' => 'textfield',
          '#title' => t('Namespace'),
          '#description' => t('The Tolgee Namespace for this export. You can override the auto-detected value here.'),
          '#default_value' => $namespace,
      ];
      return $form;
  }
}