<?php

namespace Drupal\Tests\tmgmt_tolgee_reverse_sync\Unit;

use Drupal\Tests\UnitTestCase;
use Drupal\tmgmt_tolgee_reverse_sync\EventSubscriber\TolgeeReverseSyncSubscriber;
use Drupal\core_event_dispatcher\Event\Entity\EntityUpdateEvent;
use Drupal\tmgmt\JobItemInterface;
use Drupal\tmgmt\JobInterface;
use Drupal\tmgmt\TranslatorInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\Query\QueryInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Language\LanguageInterface;

/**
 * @coversDefaultClass \Drupal\tmgmt_tolgee_reverse_sync\EventSubscriber\TolgeeReverseSyncSubscriber
 * @group tmgmt_tolgee
 */
class TolgeeReverseSyncSubscriberTest extends UnitTestCase {

  protected $subscriber;
  protected $entityTypeManager;
  protected $configFactory;
  protected $currentUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    
    // 1. Mock Container Services
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->currentUser = $this->createMock(AccountProxyInterface::class);
    
    $container = new ContainerBuilder();
    $container->set('entity_type.manager', $this->entityTypeManager);
    $container->set('config.factory', $this->configFactory);
    $container->set('current_user', $this->currentUser);
    \Drupal::setContainer($container);

    $this->subscriber = new TolgeeReverseSyncSubscriber();
  }

  /**
   * Tests onEntityUpdate logic.
   */
  public function testOnEntityUpdate() {
    // 1. Mock Event & Entity
    $event = $this->createMock(EntityUpdateEvent::class);
    $entity = $this->createMock(ContentEntityInterface::class);
    $event->method('getEntity')->willReturn($entity);
    
    $entity->method('isNew')->willReturn(FALSE);
    $entity->method('getEntityTypeId')->willReturn('node');
    $entity->method('id')->willReturn(123);

    // 2. Mock TMGMT Job Item Query
    $storage = $this->createMock(EntityStorageInterface::class);
    $query = $this->createMock(QueryInterface::class);
    
    $this->entityTypeManager->method('getStorage')->with('tmgmt_job_item')->willReturn($storage);
    $storage->method('getQuery')->willReturn($query);
    $query->method('condition')->willReturnSelf();
    $query->method('accessCheck')->willReturnSelf();
    $query->method('execute')->willReturn([1 => '1']); // Found one job item

    // 3. Mock Job Item Loading
    $jobItem = $this->createMock(JobItemInterface::class);
    $storage->method('loadMultiple')->willReturn([1 => $jobItem]);

    // 4. Mock Translations Loop
    $langFR = $this->createMock(LanguageInterface::class);
    $entity->method('getTranslationLanguages')->willReturn(['fr' => $langFR]);
    $entity->method('hasTranslation')->with('fr')->willReturn(TRUE);
    
    $translatedEntity = $this->createMock(ContentEntityInterface::class);
    $entity->method('getTranslation')->with('fr')->willReturn($translatedEntity);
    
    // CRITICAL: Simulate local change detected
    $translatedEntity->method('hasTranslationChanges')->willReturn(TRUE);

    // 5. Mock Job & Translator Checks
    $job = $this->createMock(JobInterface::class);
    $jobItem->method('getJob')->willReturn($job);
    
    $config = $this->createMock(ImmutableConfig::class);
    $config->method('get')->with('enable_reverse_sync')->willReturn(TRUE);
    $this->configFactory->method('get')->willReturn($config);

    $translator = $this->createMock(TranslatorInterface::class);
    $translator->method('getPluginId')->willReturn('tolgee');
    $job->method('getTranslator')->willReturn($translator);
    $job->method('getTargetLangcode')->willReturn('fr');

    // 6. Mock State Check (Must be ACCEPTED to trigger logic)
    $jobItem->method('getState')->willReturn(JobItemInterface::STATE_ACCEPTED);
    
    // 7. Mock Data & Content comparison
    // Structure says we have 'body'.
    $jobItem->method('getData')->willReturn([
        'body' => [
            'value' => [ // TMGMT structure can be deep
                '#text' => 'Old Value',
                '#translation' => ['#text' => 'Old Translation']
            ]
        ]
    ]);
    
    // Entity return 'New Translation'
    $fieldList = $this->createMock(\Drupal\Core\Field\FieldItemListInterface::class);
    $fieldItem = $this->createMock(\Drupal\Core\Field\FieldItemInterface::class);

    // Mock accessing the field chain for getLocalValue
    $translatedEntity->method('hasField')->willReturn(TRUE);
    $translatedEntity->method('get')->willReturn($fieldList);
    $fieldList->method('offsetExists')->willReturn(TRUE);
    $fieldList->method('get')->willReturn($fieldItem);
    // Mock getting the 'value' property
    $propertyItem = $this->createMock(\Drupal\Core\TypedData\TypedDataInterface::class);
    $propertyItem->method('getValue')->willReturn('New Translation');
    $fieldItem->method('get')->with('value')->willReturn($propertyItem);


    // 8. EXPECTATIONS
    // We expect setState(REVIEW) first
    $jobItem->expects($this->once())->method('setState')->with(JobItemInterface::STATE_REVIEW);
    
    // We expect updateData to be called via forceUpdate logic
    // We do NOT expect addTranslatedData anymore for the force flow.
    $jobItem->expects($this->once())->method('updateData');
    
    // We expect SAVE
    $jobItem->expects($this->once())->method('save');

    // Run
    $this->subscriber->onEntityUpdate($event);
  }
}
