<?php

namespace Drupal\tmgmt_tolgee\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\tmgmt\JobInterface;

/**
 * Service to handle synchronization (Cron) for Tolgee Jobs.
 */
class TolgeeSynchronizer {

  protected $entityTypeManager;
  protected $logger;
  protected $messenger;

  public function __construct(EntityTypeManagerInterface $entity_type_manager, LoggerChannelFactoryInterface $logger_factory, MessengerInterface $messenger) {
    $this->entityTypeManager = $entity_type_manager;
    $this->logger = $logger_factory->get('tmgmt_tolgee');
    $this->messenger = $messenger;
  }

  /**
   * Cron Handler.
   */
  public function runCron() {
    $storage = $this->entityTypeManager->getStorage('tmgmt_job');
    $limit = 20; // Batch limit per cron run to prevent timeouts

    // 1. Auto-push NEW jobs (Unprocessed)
    $query = $storage->getQuery()
      ->condition('state', JobInterface::STATE_UNPROCESSED)
      ->range(0, $limit)
      ->accessCheck(FALSE);
    $ids = $query->execute();
    if ($ids) {
      $unprocessed = $storage->loadMultiple($ids);
      $this->processCronBatch($unprocessed, 'push');
    }

    // 2. Auto-pull ACTIVE jobs
    $query = $storage->getQuery()
      ->condition('state', JobInterface::STATE_ACTIVE)
      ->range(0, $limit)
      ->accessCheck(FALSE);
    $ids = $query->execute();
    if ($ids) {
        $active = $storage->loadMultiple($ids);
        $this->processCronBatch($active, 'pull');
    }

    // 3. Sync back FINISHED jobs (Pull Only)
    // We limit this to 20 as well, assuming frequent cron runs will eventually cover all.
    // Optimization: We could sort by 'changed' ASC to prioritize checking old jobs?
    // For now, standard range is sufficient as the 'fetchTranslation' uses ETag (304) for speed.
    $query = $storage->getQuery()
      ->condition('state', JobInterface::STATE_FINISHED)
      ->range(0, $limit)
      ->accessCheck(FALSE);
    $ids = $query->execute();
    if ($ids) {
        $finished = $storage->loadMultiple($ids);
        $this->processCronBatch($finished, 'pull');
    }
  }

  protected function processCronBatch(array $jobs, $op) {
    foreach ($jobs as $job) {
      if (!$job->hasTranslator() || $job->getTranslator()->getPluginId() !== 'tolgee') continue;
      if (!$job->getTranslator()->getSetting('auto_sync')) continue;

      try {
        if ($op === 'push') {
          $job->requestTranslation();
          $this->logger->info('Cron: Auto-pushed NEW job @id.', ['@id' => $job->id()]);
        }
        elseif ($op === 'pull') {
          $job->getTranslator()->getPlugin()->fetchTranslation($job);
        }
      } catch (\Exception $e) {
        $this->logger->error('Cron @op failed for job @id: @msg', ['@op' => $op, '@id' => $job->id(), '@msg' => $e->getMessage()]);
      }
    }
  }

  /**
   * Helper to set/unset a value in the Job settings field.
   */
  protected function updateJobSetting(JobInterface $job, $key, $value) {
      $settings_field = $job->get('settings');
      $values = $settings_field->isEmpty() ? [] : $settings_field->first()->getValue();
      
      if ($value === NULL) {
          if (isset($values[$key])) unset($values[$key]);
      } else {
          $values[$key] = $value;
      }
      
      $job->set('settings', [$values]);
  }
}