<?php

declare(strict_types=1);

namespace Drupal\tmgmt_tolgee_reverse_sync\EventSubscriber;

use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\core_event_dispatcher\EntityHookEvents;
use Drupal\core_event_dispatcher\Event\Entity\EntityUpdateEvent;
use Drupal\tmgmt\Entity\Job;
use Drupal\tmgmt\JobItemInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Reacts to entity updates to flag TMGMT translations for review.
 *
 * This subscriber detects changes in source entities (nodes, terms, etc.) and
 * compares them against the last known TMGMT state. If a manual change is
 * detected in a translation, it flags the corresponding TMGMT Job Item as
 * 'Needs Review' and updates it with the local content, enabling a reverse push.
 */
class TolgeeReverseSyncSubscriber implements EventSubscriberInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * Constructs a TolgeeReverseSyncSubscriber object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   */
  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    ConfigFactoryInterface $configFactory,
    AccountProxyInterface $currentUser,
    TimeInterface $time
  ) {
    $this->entityTypeManager = $entityTypeManager;
    $this->configFactory = $configFactory;
    $this->currentUser = $currentUser;
    $this->time = $time;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      EntityHookEvents::ENTITY_UPDATE => 'onEntityUpdate',
    ];
  }

  /**
   * Reverts Job Item status to 'Needs Review' when source entity is modified.
   *
   * @param \Drupal\core_event_dispatcher\Event\Entity\EntityUpdateEvent $event
   *   The entity update event.
   */
  public function onEntityUpdate(EntityUpdateEvent $event): void {
    $entity = $event->getEntity();

    if (!($entity instanceof ContentEntityInterface) || $entity->isNew()) {
      return;
    }

    $storage = $this->entityTypeManager->getStorage('tmgmt_job_item');
    $result = $storage->getQuery()
      ->condition('plugin', 'content')
      ->condition('item_type', $entity->getEntityTypeId())
      ->condition('item_id', $entity->id())
      ->accessCheck(FALSE)
      ->execute();

    if (empty($result)) {
      return;
    }

    /** @var \Drupal\tmgmt\JobItemInterface[] $all_job_items */
    $all_job_items = $storage->loadMultiple($result);

    // 1. Iterate ALL languages on the entity to find the modified one.
    foreach ($entity->getTranslationLanguages() as $langcode => $language) {
      if (!$entity->hasTranslation($langcode)) {
        continue;
      }

      // Get the specific translation object (Guarantees we read correct values).
      $translated_entity = $entity->getTranslation($langcode);

      // 2. Check if THIS translation changed.
      if ($translated_entity->hasTranslationChanges()) {

        foreach ($all_job_items as $job_item) {
          $job = $job_item->getJob();

          // ================================================================
          // CHECK GLOBAL CONFIGURATION
          // ================================================================
          $config = $this->configFactory->get('tmgmt_tolgee_reverse_sync.settings');

          // If the global "Enable" switch is off, do nothing.
          if (!$config->get('enable_reverse_sync')) {
            continue;
          }

          // Verify this job is actually using the Tolgee translator.
          $translator = $job->getTranslator();
          if (!$translator || $translator->getPluginId() !== 'tolgee') {
            continue;
          }
          // ================================================================

          if ($job->getTargetLangcode() !== $langcode) {
            continue;
          }

          // 3. Status Check: Only sync back if the job is effectively "Done" OR "In Review".
          // We must handle STATE_REVIEW to allow "Latest Edit Wins" (A -> B -> C scenarios).
          $state = $job_item->getState();
          if ($state == JobItemInterface::STATE_ACCEPTED || $state == JobItemInterface::STATE_REVIEW) {

            // Get the structure TMGMT expects.
            $tmgmt_structure = $job_item->getData();

            // 4. Build a CLEAN translation array (Only Keys + Text).
            // We do NOT modify $tmgmt_structure directly. We build a new array.
            $clean_translation_data = [];
            $has_changes = $this->buildCleanTranslationData($translated_entity, $tmgmt_structure, $clean_translation_data);

            if ($has_changes) {
              $username = $this->currentUser->getAccountName() ?: 'System/Anonymous';
              $msg = "Local update detected in '$langcode' by user: $username. Flagged for review.";

              // IMPORTANT: Set state to REVIEW *before* adding data.
              $job_item->setState(JobItemInterface::STATE_REVIEW);

              // 6. Persist using Force Update.
              // We bypass addTranslatedData's safety checks (which prevent overwriting local edits)
              // by writing directly to the data array recursively.
              $this->forceUpdateTranslationData($job_item, $clean_translation_data);

              $job_item->addMessage($msg, [], 'warning');
              $job_item->save();

              if ($job->getState() == Job::STATE_FINISHED) {
                $job->setState(Job::STATE_ACTIVE);
                $job->save();
              }
            }
          }
        }
      }
    }
  }

  /**
   * Recursively builds a clean data array for TMGMT by comparing values.
   *
   * This method iterates through the TMGMT data structure, fetches the current
   * values from the entity, and compares them. If a difference is found, it
   * adds the new value to the output array.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The translated entity to read values from.
   * @param array $structure
   *   The TMGMT data structure defining which fields to check.
   * @param array $output
   *   The clean output array being built (by reference).
   * @param array $key_chain
   *   Internal recursion tracker for array depth.
   *
   * @return bool
   *   TRUE if any changes were detected and added to $output, FALSE otherwise.
   */
  private function buildCleanTranslationData(ContentEntityInterface $entity, array $structure, array &$output, array $key_chain = []): bool {
    $changes_found = FALSE;

    foreach ($structure as $key => $element) {
      if (isset($element['#text']) && isset($element['#translation']['#text'])) {

        // 1. Get Fresh Value from Entity.
        $local_value = $this->getLocalValue($entity, $key, $key_chain);
        $tmgmt_value = $element['#translation']['#text'];

        // 2. Compare (Trim to ignore whitespace diffs).
        if (trim((string) $local_value) !== trim((string) $tmgmt_value)) {

          // 3. Add to Output Array using NestedArray.
          $parents = array_merge($key_chain, [$key]);

          NestedArray::setValue($output, $parents, [
            '#text' => $local_value,
            '#origin' => 'local',
          ]);

          $changes_found = TRUE;
        }
      }
      elseif (is_array($element)) {
        $new_chain = $key_chain;
        $new_chain[] = $key;
        if ($this->buildCleanTranslationData($entity, $element, $output, $new_chain)) {
          $changes_found = TRUE;
        }
      }
    }
    return $changes_found;
  }

  /**
   * Helper to extract a field value from the entity using the key chain.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to read from.
   * @param string|int $key
   *   The current key in the structure loop.
   * @param array $key_chain
   *   The path to the current element.
   *
   * @return string|mixed
   *   The value found in the entity, or an empty string if not found.
   */
  private function getLocalValue(ContentEntityInterface $entity, $key, array $key_chain) {
    $field_name = $key_chain[0] ?? $key;
    if (!$entity->hasField($field_name)) {
      return '';
    }

    // Determine Delta: TMGMT usually puts fields in [field][delta][property].
    $delta = (isset($key_chain[1]) && is_numeric($key_chain[1])) ? $key_chain[1] : (is_numeric($key) ? $key : 0);
    $property = (isset($key_chain[2])) ? $key_chain[2] : (!is_numeric($key) ? $key : 'value');

    $local_value = '';
    $field_list = $entity->get($field_name);

    if ($field_list->offsetExists($delta)) {
      $item = $field_list->get($delta);
      try {
        $local_value = $item->get($property)->getValue();
      }
      catch (\Exception $e) {
        // Fallback for some field types.
        if (isset($item->$property)) {
          $local_value = $item->$property;
        }
      }
    }

    return $local_value;
  }

  /**
   * Helper to force an update of translation data.
   *
   * @param \Drupal\tmgmt\JobItemInterface $job_item
   *   The job item to update.
   * @param array $clean_data
   *   The structure containing the keys we want to reset.
   */
  private function forceUpdateTranslationData(JobItemInterface $job_item, array $clean_data) {
    // Recursive update of data items.
    $this->forceUpdateRecursive($job_item, $clean_data, []);
  }

  /**
   * Recursive helper for force update.
   */
  private function forceUpdateRecursive(JobItemInterface $job_item, array $data, array $key_chain) {
    foreach ($data as $key => $value) {
      if (isset($value['#text'])) {
        // Found a leaf. Construct the TMGMT key array.
        $full_key = array_merge($key_chain, [$key]);

        // Ensure text is never null to avoid deprecated warnings in TMGMT.
        $text_val = isset($value['#text']) ? (string) $value['#text'] : '';

        // DIRECT UPDATE: We overwrite whatever is there with the new local value.
        // We do *not* use addTranslatedData because it tries to be smart about revisions.
        // We simply want "Latest Local Edit Wins".
        $job_item->updateData($full_key, [
          '#translation' => [
            '#text' => $text_val,
            '#origin' => 'local',
            '#timestamp' => $this->time->getRequestTime(),
          ],
        ]);
      }
      elseif (is_array($value)) {
        $this->forceUpdateRecursive($job_item, $value, array_merge($key_chain, [$key]));
      }
    }
  }

}