<?php

declare(strict_types=1);

namespace Drupal\tmgmt_tolgee_reverse_sync;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\key\KeyRepositoryInterface;
use Drupal\tmgmt\Data;
use Drupal\tmgmt\JobItemInterface;
use Drupal\tmgmt_tolgee\Plugin\tmgmt\Translator\TolgeeTranslator;
use GuzzleHttp\ClientInterface;

/**
 * Service to push single translation items back to Tolgee.
 *
 * This service is used by the Reverse Sync submodule to propagate local
 * translation changes (made in Drupal) effectively back to the Tolgee platform.
 */
class TolgeePusher {

  /**
   * Constructs a TolgeePusher object.
   *
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   The HTTP client.
   * @param \Drupal\key\KeyRepositoryInterface $keyRepository
   *   The key repository service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory.
   * @param \Drupal\tmgmt\Data $tmgmtData
   *   The TMGMT data service.
   */
  public function __construct(
    private ClientInterface $httpClient,
    private KeyRepositoryInterface $keyRepository,
    private LoggerChannelFactoryInterface $loggerFactory,
    private Data $tmgmtData
  ) {}

  /**
   * Pushes a single job item to Tolgee.
   *
   * @param \Drupal\tmgmt\JobItemInterface $item
   *   The job item containing the translation update.
   */
  public function pushItem(JobItemInterface $item): void {
    $translator = $item->getJob()->getTranslator();
    $plugin = $translator->getPlugin();

    if (!$plugin instanceof TolgeeTranslator) {
      return;
    }

    // Use the plugin's method to get the correct host (e.g. tolgee.viappi.com).
    $endpoint = $plugin->getProjectUrl($translator) . '/translations';

    $keyId = $translator->getSetting('api_key');
    $keyEntity = $this->keyRepository->getKey($keyId);
    if (!$keyEntity) {
      return;
    }
    $apiKey = $keyEntity->getKeyValue();
    if (!$apiKey) {
      return;
    }

    $targetLang = $item->getJob()->getTargetLanguage()->getId();

    // Flatten the TMGMT data to find 'name][0][value'.
    $flattened = $this->tmgmtData->flatten($item->getData());

    foreach ($flattened as $keyPath => $values) {
      // Look for the translated text string.
      $text = $values['#translation']['#text'] ?? $values['#translation'] ?? NULL;

      if (is_string($text) && !empty($text)) {

        // Determine Namespace using Translator Logic.
        // This ensures the push targets the correct "folder" in Tolgee.
        $namespace = 'drupal';
        $determined = $plugin->determineNamespace($item->getJob());
        if (!empty($determined)) {
          $namespace = $determined;
        }

        // Fix key naming to match what Tolgee expects (persistent).
        $persistentId = "{$item->getItemType()}:{$item->getItemId()}][{$keyPath}";

        $payload = [
          'key' => $persistentId,
          'namespace' => $namespace,
          'translations' => [$targetLang => $text],
        ];

        try {
          $this->httpClient->request('POST', $endpoint, [
            'headers' => [
              'X-API-Key' => $apiKey,
              'Content-Type' => 'application/json',
            ],
            'json' => $payload,
          ]);

        }
        catch (\Exception $e) {
          $this->loggerFactory->get('tmgmt_tolgee_reverse_sync')->error(
            'Push Failed to NS @ns: @msg', ['@ns' => $namespace, '@msg' => $e->getMessage()]
          );
        }
      }
    }
  }

}