<?php

declare(strict_types=1);

namespace Drupal\tmgmt_tolgee\Plugin\tmgmt\Translator;

use Drupal\Component\Plugin\PluginManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\State\StateInterface;
use Drupal\key\KeyRepositoryInterface;
use Drupal\tmgmt\Data;
use Drupal\tmgmt\JobInterface;
use Drupal\tmgmt\Translator\AvailableResult;
use Drupal\tmgmt\Translator\TranslatableResult;
use Drupal\tmgmt\TranslatorInterface;
use Drupal\tmgmt\TranslatorPluginBase;
use Drupal\tmgmt\TMGMTException;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Tolgee implementation of the TranslatorPluginBase.
 *
 * @TranslatorPlugin(
 *   id = "tolgee",
 *   label = @Translation("Tolgee Translator"),
 *   description = @Translation("Integrates with Tolgee API for shared translations."),
 *   ui = "Drupal\tmgmt_tolgee\TolgeeTranslatorUi",
 *   default_settings = {
 *     "base_url" = "https://app.tolgee.io",
 *     "api_key" = "",
 *     "project_id" = "",
 *     "namespace" = "drupal",
 *     "auto_accept" = FALSE,
 *     "auto_sync" = FALSE,
 *     "namespace_mapping" = {},
 *     "import_options" = {
 *       "tag_source" = TRUE,
 *     }
 *   },
 *   logo = "icons/tolgee.svg",
 * )
 */
class TolgeeTranslator extends TranslatorPluginBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a TolgeeTranslator object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \GuzzleHttp\ClientInterface $client
   *   The HTTP client.
   * @param \Drupal\key\KeyRepositoryInterface $keyRepository
   *   The key repository.
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   The file system service.
   * @param \Drupal\Component\Plugin\PluginManagerInterface $formatManager
   *   The format manager.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory.
   * @param \Drupal\tmgmt\Data $dataService
   *   The TMGMT data service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected ClientInterface $client,
    protected KeyRepositoryInterface $keyRepository,
    protected FileSystemInterface $fileSystem,
    protected PluginManagerInterface $formatManager,
    protected StateInterface $state,
    protected LoggerChannelFactoryInterface $loggerFactory,
    protected Data $dataService,
    protected EntityTypeManagerInterface $entityTypeManager
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('http_client'),
      $container->get('key.repository'),
      $container->get('file_system'),
      $container->get('plugin.manager.tmgmt_file.format'),
      $container->get('state'),
      $container->get('logger.factory'),
      $container->get('tmgmt.data'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * Retrieves the project URL from configuration.
   *
   * @param \Drupal\tmgmt\TranslatorInterface $translator
   *   The translator configuration entity.
   *
   * @return string
   *   The constructed project URL (e.g., base_url/v2/projects/ID).
   */
  public function getProjectUrl(TranslatorInterface $translator): string {
    $base_url = rtrim((string) $translator->getSetting('base_url'), '/');
    $project_id = $translator->getSetting('project_id');
    if (!empty($project_id)) {
      return $base_url . '/v2/projects/' . $project_id;
    }
    return $base_url . '/v2/projects';
  }

  /**
   * {@inheritdoc}
   */
  public function checkAvailable(TranslatorInterface $translator) {
    $api_key = $translator->getSetting('api_key');
    if (!$api_key) {
      return AvailableResult::no($this->t('@translator is not available. Make sure it is properly configured.', ['@translator' => $translator->label()]));
    }

    $key_entity = $this->keyRepository->getKey($api_key);
    if ($key_entity) {
      $key_value = $key_entity->getKeyValue();
      if (is_string($key_value) && str_starts_with($key_value, 'tgpat_') && empty($translator->getSetting('project_id'))) {
        return AvailableResult::no($this->t('A Project ID is required when using a Personal Access Token (tgpat_).'));
      }
    }

    return AvailableResult::yes();
  }

  /**
   * {@inheritdoc}
   */
  public function checkTranslatable(TranslatorInterface $translator, JobInterface $job) {
    return TranslatableResult::yes();
  }

  /**
   * {@inheritdoc}
   */
  public function hasCheckoutSettings(JobInterface $job) {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function requestTranslation(JobInterface $job) {
    /** @var \Drupal\tmgmt_file\Format\FormatManager $format_manager */
    $converter = $this->formatManager->createInstance('xlf');
    $content = $converter->export($job);

    foreach ($job->getItems() as $item) {
      $transient_id = $item->id();
      $persistent_id = $item->getItemType() . ':' . $item->getItemId();
      $content = str_replace('id="' . $transient_id . '][', 'id="' . $persistent_id . '][', $content);
      $content = str_replace('resname="' . $transient_id . '][', 'resname="' . $persistent_id . '][', $content);
    }

    $job_ns = $job->getSetting('namespace');
    $global_ns = $job->getTranslator()->getSetting('namespace');

    if (empty($job_ns) || $job_ns === $global_ns) {
      $detected = $this->determineNamespace($job);
      if (!empty($detected)) {
        $namespace = $detected;
      }
      else {
        $namespace = $job_ns ?: $global_ns;
      }
    }
    else {
      $namespace = $job_ns;
    }

    $key_id = $job->getTranslator()->getSetting('api_key');
    $key_entity = $this->keyRepository->getKey($key_id);
    if (!$key_entity) {
      throw new TMGMTException('Tolgee API Key not configured.');
    }
    $api_key = $key_entity->getKeyValue();

    $project_url = $this->getProjectUrl($job->getTranslator());
    $endpoint = $project_url . '/single-step-import';
    $filename = 'job_' . $job->id() . '.xlf';
    $import_options = $job->getTranslator()->getSetting('import_options');
    $force_mode = $import_options['force_mode'] ?? 'OVERRIDE';

    $params = [
      'format' => 'XLIFF',
      'forceMode' => $force_mode,
      'overrideKeyDescriptions' => TRUE,
      'convertPlaceholdersToIcu' => FALSE,
      'createNewKeys' => TRUE,
      'fileMappings' => empty($namespace) ? [['fileName' => $filename]] : [['fileName' => $filename, 'namespace' => $namespace]],
    ];

    $tags = [];
    $tag_source = $import_options['tag_source'] ?? TRUE;
    if ($tag_source) {
      $tags[] = 'source-' . $job->getSourceLanguage()->getId();
    }
    $custom_tags = $job->getSetting('tags');
    if (!empty($custom_tags)) {
      $tags = array_merge($tags, array_map('trim', explode(',', $custom_tags)));
    }
    if (!empty($tags)) {
      $params['tagNewKeys'] = array_values(array_unique($tags));
    }

    try {
      $this->client->request('POST', $endpoint, [
        'headers' => ['X-API-Key' => $api_key, 'Accept' => 'application/json'],
        'multipart' => [
          ['name' => 'files', 'contents' => $content, 'filename' => $filename],
          [
            'name' => 'params',
            'contents' => json_encode($params),
            'headers' => ['Content-Type' => 'application/json'],
          ],
        ],
      ]);

      $job->submitted('Job submitted to Tolgee.');

    }
    catch (\Exception $e) {
      $msg = $e->getMessage();
      if (strpos($msg, 'ConstraintViolationException') !== FALSE || strpos($msg, 'duplicate key') !== FALSE) {
        $this->loggerFactory->get('tmgmt_tolgee')->warning('Job @id: Duplicate key detected. Assuming success.', ['@id' => $job->id()]);
        $job->submitted('Job submitted (Duplicate key handling triggered).');
      }
      else {
        $job->rejected('Tolgee API Error: ' . $msg);
        throw new TMGMTException('Tolgee API Error: ' . $msg, [], $e->getCode(), $e);
      }
    }

    $mapped_count = 0;
    foreach ($job->getItems() as $item) {
      $persistent_id = $item->getItemType() . ':' . $item->getItemId();
      try {
        $search_response = $this->client->request('GET', $project_url . '/keys', [
          'headers' => ['X-API-Key' => $api_key, 'Accept' => 'application/json'],
          'query' => ['search' => $persistent_id, 'size' => 50],
        ]);
        $data = json_decode($search_response->getBody()->getContents(), TRUE);
        if (!empty($data['_embedded']['keys'])) {
          foreach ($data['_embedded']['keys'] as $tolgee_key) {
            if (str_starts_with($tolgee_key['name'], $persistent_id)) {
              $item->addRemoteMapping(NULL, $tolgee_key['id'], ['remote_key' => $tolgee_key['name']]);
              $mapped_count++;
            }
          }
          $item->save();
        }
      }
      catch (\Exception $e) {
        // Ignore errors during key mapping helper.
      }
    }
    if ($mapped_count > 0) {
      $job->addMessage('Mapped @count remote keys to local job items.', ['@count' => $mapped_count]);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function fetchTranslation(JobInterface $job) {
    $key_id = $job->getTranslator()->getSetting('api_key');
    $key_entity = $this->keyRepository->getKey($key_id);
    if (!$key_entity) {
      throw new TMGMTException('Tolgee API Key not configured.');
    }
    $api_key = $key_entity->getKeyValue();

    $project_url = $this->getProjectUrl($job->getTranslator());
    $target_lang = $job->getTargetLangcode();

    // 1. COLLECT IDS (With Batching for Scalability).
    $tjiids = $job->getItemIds();

    // Safety: If job has > 200 items, we paginate.
    $batch_size = 200;
    if (count($tjiids) > $batch_size) {
      $offset = isset($job->getSettings()['sync_offset']) ? $job->getSettings()['sync_offset'] : 0;

      // Wrap around.
      if ($offset >= count($tjiids)) {
        $offset = 0;
      }

      $batch_tjiids = array_slice($tjiids, $offset, $batch_size);

      // Next offset.
      $next_offset = $offset + $batch_size;
      if ($next_offset >= count($tjiids)) {
        $next_offset = 0;
      }

      // Save offset.
      $settings = $job->get('settings')->getValue();
      // Handle delta 0 correctly.
      if (empty($settings)) {
        $settings = [['sync_offset' => $next_offset]];
      }
      else {
        $settings[0]['sync_offset'] = $next_offset;
      }
      $job->set('settings', $settings);
      $job->save();

      $job_items = $this->entityTypeManager->getStorage('tmgmt_job_item')->loadMultiple($batch_tjiids);
    }
    else {
      $job_items = $job->getItems();
    }

    $ids_to_export = [];
    foreach ($job_items as $item) {
      $mappings = $item->getRemoteMappings();
      if (!empty($mappings)) {
        foreach ($mappings as $mapping) {
          if (!$mapping->remote_identifier_1->isEmpty()) {
            $ids_to_export[] = (int) $mapping->remote_identifier_1->value;
          }
        }
      }
    }

    $ids_to_export = array_unique($ids_to_export);
    sort($ids_to_export);
    if (empty($ids_to_export)) {
      $job->addMessage('No remote keys found in Tolgee.', 'warning');
      return;
    }

    // 2. EXPORT (POST).
    try {
      $export_endpoint = $project_url . '/export';

      $ids_hash = md5(serialize($ids_to_export));
      $etag_key = 'tmgmt_tolgee.etag.' . $job->id() . '.' . $ids_hash;
      $etag = $this->state->get($etag_key);

      $options = [
        'headers' => ['X-API-Key' => $api_key, 'Accept' => 'application/json, application/zip'],
        'json' => [
          'format' => 'JSON',
          'languages' => [$target_lang],
          'zip' => FALSE,
          'structureDelimiter' => '',
          'filterState' => ['TRANSLATED', 'REVIEWED'],
          'filterKeyId' => array_values($ids_to_export),
        ],
      ];

      if ($etag) {
        $options['headers']['If-None-Match'] = $etag;
      }

      $response = $this->client->request('POST', $export_endpoint, $options);

      // 304 Logic: API says "Nothing changed" based on ETag.
      if ($response->getStatusCode() === 304) {
        $this->updateJobStatus($job);
        return;
      }

      // SAVE THE ETAG (Crucial to stop the loop!).
      if ($response->hasHeader('ETag')) {
        $this->state->set($etag_key, $response->getHeaderLine('ETag'));
      }

      $content = $response->getBody()->getContents();

      // JSON Mode: Decode directly.
      $combined_data = json_decode($content, TRUE);
      if (!is_array($combined_data)) {
        // Fallback/Error if not array.
        $combined_data = [];
      }
      // Note: In JSON mode for single language, structure is usually flat or nested based on keys.
      // But Tolgee export with multiple keys usually returns the structure we want.

      if (empty($combined_data)) {
        $this->loggerFactory->get('tmgmt_tolgee')->info('Job @id: Tolgee returned no data (possibly all filtered by State or 304).', ['@id' => $job->id()]);
        $this->updateJobStatus($job);
        return;
      }

      // 3. REVERSE MAPPING.
      $id_map = [];
      foreach ($job->getItems() as $item) {
        $persistent_id = $item->getItemType() . ':' . $item->getItemId();
        $id_map[$persistent_id] = $item->id();
      }

      $imported_data = [];
      foreach ($combined_data as $key => $value) {
        if (empty($key) || !is_string($value)) {
          continue;
        }
        $parts = explode('][', $key, 2);
        if (count($parts) === 2) {
          $persistent_prefix = $parts[0];
          $suffix = $parts[1];
          if (isset($id_map[$persistent_prefix])) {
            $tjiid = $id_map[$persistent_prefix];
            $original_key = $tjiid . '][' . $suffix;
            $imported_data[$original_key]['#text'] = $value;
            $imported_data[$original_key]['#origin'] = 'Tolgee';
          }
        }
      }

      if (!empty($imported_data)) {
        $data = $this->dataService->unflatten($imported_data);

        // =====================================================================
        // PHANTOM FILTER: Checks if text ACTUALLY changed before updating
        // =====================================================================
        foreach ($data as $tjiid => $item_data) {
          $job_item = $this->entityTypeManager->getStorage('tmgmt_job_item')->load($tjiid);
          if ($job_item) {
            $current_data = $job_item->getData();
            // Filter out keys where the translation is identical.
            $real_changes = $this->filterUnchangedData($item_data, $current_data);

            if (empty($real_changes)) {
              unset($data[$tjiid]);
            }
            else {
              $data[$tjiid] = $real_changes;

              // FORCE REVIEW STATE:
              // Since we are reopening finished jobs, TMGMT might not automatically demote
              // an 'Accepted' item back to 'Review'. We force it here unless auto-accept is on.
              if (!$job->getTranslator()->getSetting('auto_accept')) {
                $job_item->setState(\Drupal\tmgmt\JobItemInterface::STATE_REVIEW);
                $job_item->save();
              }
            }
          }
        }

        // If data is empty after filtering, it means it was a Phantom Update.
        // We SAVED the ETag above, so next time we get 304.
        // We RETURN here to prevent re-opening the job.
        if (empty($data)) {
          $this->loggerFactory->get('tmgmt_tolgee')->notice('Job @id: 200 OK but content identical. Ignored.', ['@id' => $job->id()]);
          $this->updateJobStatus($job);
          return;
        }
        // =====================================================================

        // Allow updating finished jobs for Continuous Translation
        // If we found changes, we must ensure the job is open to accept them.
        if ($job->isFinished()) {
          $job->setState(JobInterface::STATE_ACTIVE);
          $job->save();
        }

        $job->addTranslatedData($data);
        $job->addMessage('Successfully imported translation from Tolgee.');

        $job->save();
        $this->updateJobStatus($job);
      }

    }
    catch (\Exception $e) {
      throw new TMGMTException('Tolgee Fetch Error: ' . $e->getMessage(), [], $e->getCode(), $e);
    }
  }

  /**
   * Helper: Recursively compares incoming data against existing data.
   *
   * @param array $incoming
   *   The incoming data structure from Tolgee.
   * @param array $existing
   *   The existing data structure in TMGMT.
   *
   * @return array
   *   The subset of $incoming containing actual changes.
   */
  private function filterUnchangedData(array $incoming, array $existing): array {
    $diff = [];
    foreach ($incoming as $key => $value) {
      if ($key === '#text') {
        $new_text = trim((string) $value);

        // 2. CHANGE DETECTION: Compare against the existing TRANSLATION.
        $old_trans = '';
        if (isset($existing['#translation']['#text'])) {
          $old_trans = trim((string) $existing['#translation']['#text']);
        }
        elseif (isset($existing['#translation']) && is_string($existing['#translation'])) {
          $old_trans = trim((string) $existing['#translation']);
        }

        if ($new_text !== $old_trans) {
          $diff[$key] = $value;
        }
      }
      elseif (is_array($value) && strpos((string) $key, '#') !== 0) {
        // RECURSION:
        $existing_element = isset($existing[$key]) ? $existing[$key] : [];
        $sub_diff = $this->filterUnchangedData($value, $existing_element);
        if (!empty($sub_diff)) {
          $diff[$key] = $sub_diff;
        }
      }
      // IGNORE ALL OTHER KEYS (like #origin, #timestamp, etc.)
    }
    return $diff;
  }

  /**
   * Updates Job Status: Auto-accepts items and Finishes job.
   *
   * @param \Drupal\tmgmt\JobInterface $job
   *   The job to update.
   */
  private function updateJobStatus(JobInterface $job): void {
    $items = $job->getItems();
    $ids = array_keys($items);

    if (empty($ids)) {
      return;
    }

    // 1. AUTO-ACCEPT STEP (Only runs if setting is enabled).
    if ($job->getTranslator()->getSetting('auto_accept')) {
      $storage = $this->entityTypeManager->getStorage('tmgmt_job_item');
      $storage->resetCache($ids);
      /** @var \Drupal\tmgmt\JobItemInterface[] $fresh_items */
      $fresh_items = $storage->loadMultiple($ids);

      foreach ($fresh_items as $item) {
        // Skip local edits from auto-acceptance.
        if ($this->hasLocalOrigin($item->getData())) {
          continue;
        }

        if ($item->isNeedsReview() || ($item->getState() == \Drupal\tmgmt\JobItemInterface::STATE_ACTIVE && !empty($item->getData()))) {
          try {
            $item->acceptTranslation();
          }
          catch (\Exception $e) {
            $this->loggerFactory->get('tmgmt_tolgee')->error('Item accept failed: @msg', ['@msg' => $e->getMessage()]);
          }
        }
      }
    }
  }

  /**
   * Recursive Helper: Checks if data has 'local' origin tag.
   *
   * @param array $data
   *   The data structure to check.
   *
   * @return bool
   *   TRUE if 'local' origin tag is found, FALSE otherwise.
   */
  private function hasLocalOrigin(array $data): bool {
    foreach ($data as $element) {
      if (is_array($element)) {
        if (isset($element['#origin']) && $element['#origin'] === 'local') {
          return TRUE;
        }
        if ($this->hasLocalOrigin($element)) {
          return TRUE;
        }
      }
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedTargetLanguages(TranslatorInterface $translator, $source_language) {
    $languages = [];
    $key_id = $translator->getSetting('api_key');
    if (!$key_id) {
      return $languages;
    }

    $key_entity = $this->keyRepository->getKey($key_id);
    if (!$key_entity) {
      return $languages;
    }
    $api_key = $key_entity->getKeyValue();

    try {
      $endpoint = $this->getProjectUrl($translator) . '/languages';
      $response = $this->client->request('GET', $endpoint, [
        'headers' => ['X-API-Key' => $api_key, 'Accept' => 'application/json'],
        'query' => ['size' => 1000],
      ]);
      $data = json_decode($response->getBody()->getContents(), TRUE);
      if (isset($data['_embedded']['languages'])) {
        foreach ($data['_embedded']['languages'] as $lang) {
          $label = !empty($lang['name']) ? $lang['name'] : $lang['tag'];
          $languages[$lang['tag']] = $label;
        }
      }
    }
    catch (\Exception $e) {
      // Ignore errors during language fetch.
    }
    return $languages;
  }

  /**
   * Determines the namespace based on the job's items and configuration.
   *
   * @param \Drupal\tmgmt\JobInterface $job
   *   The translation job.
   *
   * @return string|null
   *   The determined namespace or NULL if no mapping matches.
   */
  public function determineNamespace(JobInterface $job): ?string {
    $items = $job->getItems();
    if ($item = reset($items)) {
      $current_type = strtolower($item->getItemType());
      $mapping_config = $job->getTranslator()->getSetting('namespace_mapping');
      if (is_array($mapping_config)) {
        foreach ($mapping_config as $group) {
          if (empty($group['namespace']) || empty($group['types'])) {
            continue;
          }
          if (is_array($group['types']) && in_array($current_type, $group['types'])) {
            return $group['namespace'];
          }
        }
      }
    }
    return NULL;
  }

}