<?php

declare(strict_types=1);

namespace Drupal\tmgmt_tolgee\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\tmgmt\JobInterface;

/**
 * Service to handle synchronization (Cron) for Tolgee Jobs.
 */
class TolgeeSynchronizer {

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * Constructs a TolgeeSynchronizer object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    LoggerChannelFactoryInterface $loggerFactory,
    protected MessengerInterface $messenger
  ) {
    $this->logger = $loggerFactory->get('tmgmt_tolgee');
  }

  /**
   * Cron Handler.
   */
  public function runCron(): void {
    $storage = $this->entityTypeManager->getStorage('tmgmt_job');
    $limit = 20;

    // 1. Auto-push NEW jobs (Unprocessed).
    $query = $storage->getQuery()
      ->condition('state', JobInterface::STATE_UNPROCESSED)
      ->range(0, $limit)
      ->accessCheck(FALSE);
    $ids = $query->execute();
    if ($ids) {
      $unprocessed = $storage->loadMultiple($ids);
      $this->processCronBatch($unprocessed, 'push');
    }

    // 2. Auto-pull ACTIVE jobs.
    $query = $storage->getQuery()
      ->condition('state', JobInterface::STATE_ACTIVE)
      ->range(0, $limit)
      ->accessCheck(FALSE);
    $ids = $query->execute();
    if ($ids) {
      $active = $storage->loadMultiple($ids);
      $this->processCronBatch($active, 'pull');
    }

    // 3. Sync back FINISHED jobs (Pull Only)
    // We limit this to 20 as well, assuming frequent cron runs will eventually cover all.
    // Optimization: We could sort by 'changed' ASC to prioritize checking old jobs?
    // For now, standard range is sufficient as the 'fetchTranslation' uses ETag (304) for speed.
    $query = $storage->getQuery()
      ->condition('state', JobInterface::STATE_FINISHED)
      ->range(0, $limit)
      ->accessCheck(FALSE);
    $ids = $query->execute();
    if ($ids) {
      $finished = $storage->loadMultiple($ids);
      $this->processCronBatch($finished, 'pull');
    }
  }

  /**
   * Helper to process a batch of jobs.
   *
   * @param array $jobs
   *   The jobs to process.
   * @param string $op
   *   The operation (push or pull).
   */
  protected function processCronBatch(array $jobs, string $op): void {
    foreach ($jobs as $job) {
      if (!$job->hasTranslator() || $job->getTranslator()->getPluginId() !== 'tolgee') {
        continue;
      }
      if (!$job->getTranslator()->getSetting('auto_sync')) {
        continue;
      }

      try {
        if ($op === 'push') {
          $job->requestTranslation();
          $this->logger->info('Cron: Auto-pushed NEW job @id.', ['@id' => $job->id()]);
        }
        elseif ($op === 'pull') {
          $job->getTranslator()->getPlugin()->fetchTranslation($job);
        }
      }
      catch (\Exception $e) {
        $this->logger->error('Cron @op failed for job @id: @msg', ['@op' => $op, '@id' => $job->id(), '@msg' => $e->getMessage()]);
      }
    }
  }

}