<?php

declare(strict_types=1);

namespace Drupal\tmgmt_tolgee;

use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Form\FormStateInterface;
use Drupal\tmgmt\JobInterface;
use Drupal\tmgmt\TranslatorPluginUiBase;
use Drupal\tmgmt_tolgee\Plugin\tmgmt\Translator\TolgeeTranslator;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\key\KeyRepositoryInterface;

/**
 * Tolgee translator UI.
 *
 * Handles the configuration form for the Tolgee translator.
 */
class TolgeeTranslatorUi extends TranslatorPluginUiBase {

  /**
   * The key repository.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected $keyRepository;

  /**
   * Constructs a new TolgeeTranslatorUi object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->keyRepository = \Drupal::service('key.repository');
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $translator = $form_state->getFormObject()->getEntity();

    // --- 1. SETUP WHITELIST (Tagify) ---
    $entity_definitions = \Drupal::entityTypeManager()->getDefinitions();
    $whitelist = [];
    foreach ($entity_definitions as $id => $definition) {
      $group = $definition->getGroup();
      if ($group != 'content' && $group != 'configuration') {
        continue;
      }
      $whitelist[] = $id;
    }
    sort($whitelist);

    $form['#attached']['library'][] = 'tmgmt_tolgee/admin.settings';
    $form['#attached']['drupalSettings']['tolgee']['whitelist'] = $whitelist;

    // --- 2. STANDARD SETTINGS ---
    $form['base_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Tolgee Base URL'),
      '#default_value' => $translator->getSetting('base_url'),
      '#required' => TRUE,
    ];
    $keys = $this->keyRepository->getKeys();
    $options = [];
    foreach ($keys as $id => $key) {
      $options[$id] = $key->label();
    }
    $form['api_key'] = [
      '#type' => 'select',
      '#title' => $this->t('Tolgee API Key'),
      '#options' => $options,
      '#default_value' => $translator->getSetting('api_key'),
      '#required' => TRUE,
      '#empty_option' => $this->t('- Select Key -'),
    ];
    $form['project_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Project ID'),
      '#default_value' => $translator->getSetting('project_id'),
    ];
    $form['auto_sync'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto Sync (Cron)'),
      '#default_value' => $translator->getSetting('auto_sync'),
    ];
    $form['enable_namespaces'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Namespace Management'),
      '#default_value' => $translator->getSetting('enable_namespaces') ?? TRUE,
    ];

    // --- 3. MAPPING TABLE UI ---
    $form['mapping_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'tolgee-mapping-wrapper'],
      '#states' => [
        'visible' => [
          ':input[name="settings[enable_namespaces]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['mapping_wrapper']['namespace_mapping'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Tolgee Namespace'),
        $this->t('Source Entity Types'),
        $this->t('Operations'),
      ],
      '#empty' => $this->t('No mappings defined. Add one below.'),
      '#attributes' => ['class' => ['tolgee-mapping-table']],
    ];

    // --- 4. ROW COUNT LOGIC ---
    // Initialize count if missing.
    if (!$form_state->has('tolgee_rows')) {
      $stored = $translator->getSetting('namespace_mapping') ?: [];
      $count = empty($stored) ? 1 : count($stored);
      $form_state->set('tolgee_rows', $count);
    }

    // Handle "Add Another" Click.
    $trigger = $form_state->getTriggeringElement();
    if ($trigger && end($trigger['#parents']) === 'add_item') {
      $count = $form_state->get('tolgee_rows') + 1;
      $form_state->set('tolgee_rows', $count);
    }

    $num_rows = $form_state->get('tolgee_rows');
    if ($num_rows < 1) {
      $num_rows = 1;
      $form_state->set('tolgee_rows', 1);
    }

    // --- 5. DATA RETRIEVAL ---
    $db_data = $translator->getSetting('namespace_mapping') ?: [];
    $db_data = array_values($db_data);

    $input = $form_state->getUserInput();
    $user_data = $input['settings']['mapping_wrapper']['namespace_mapping']
      ?? $input['mapping_wrapper']['namespace_mapping']
      ?? [];

    // --- 6. BUILD ROWS ---
    for ($i = 0; $i < $num_rows; $i++) {
      $ns_val = $user_data[$i]['namespace'] ?? $db_data[$i]['namespace'] ?? '';

      $types_raw = $user_data[$i]['types'] ?? $db_data[$i]['types'] ?? '';
      $types_val = is_array($types_raw) ? implode(',', $types_raw) : $types_raw;

      // Column 1: Namespace.
      $form['mapping_wrapper']['namespace_mapping'][$i]['namespace'] = [
        '#type' => 'textfield',
        '#default_value' => $ns_val,
        '#size' => 20,
        '#attributes' => ['placeholder' => 'e.g. navigation'],
      ];

      // Column 2: Types (Tagify).
      $form['mapping_wrapper']['namespace_mapping'][$i]['types'] = [
        '#type' => 'textfield',
        '#default_value' => $types_val,
        '#maxlength' => 2048,
        '#attributes' => [
          'class' => ['js-tolgee-source-types'],
          'placeholder' => $this->t('Select types...'),
          'style' => 'display: none;',
        ],
      ];

      // Column 3: Remove Button.
      $form['mapping_wrapper']['namespace_mapping'][$i]['remove'] = [
        '#type' => 'submit',
        '#value' => '×',
        '#name' => 'tolgee_remove_' . $i,
        '#submit' => [[$this, 'removeOne']],
        '#limit_validation_errors' => [],
        '#ajax' => [
          'callback' => [$this, 'addmoreCallback'],
          'wrapper' => 'tolgee-mapping-wrapper',
        ],
        '#attributes' => [
          'class' => ['button', 'button--small'],
          'title' => $this->t('Remove this row'),
          'style' => 'border-radius: 50%; width: 30px; height: 30px; padding: 0; line-height: 28px; text-align: center;',
        ],
      ];
    }

    $form['mapping_wrapper']['add_item'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add another mapping'),
      '#submit' => [[$this, 'addOne']],
      '#limit_validation_errors' => [],
      '#ajax' => [
        'callback' => [$this, 'addmoreCallback'],
        'wrapper' => 'tolgee-mapping-wrapper',
      ],
    ];

    // --- 7. REMAINING FIELDS ---
    $form['namespace'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default Namespace'),
      '#default_value' => $translator->getSetting('namespace'),
      '#states' => [
        'visible' => [
          ':input[name="settings[enable_namespaces]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['import_options'] = [
      '#type' => 'details',
      '#title' => $this->t('Import Options'),
      '#open' => TRUE,
    ];
    $settings = $translator->getSettings();
    $form['import_options']['tag_source'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Tag with Source Language ID'),
      '#default_value' => $settings['import_options']['tag_source'] ?? TRUE,
    ];
    $form['import_options']['force_mode'] = [
      '#type' => 'radios',
      '#title' => $this->t('Conflict Resolution (Force Mode)'),
      '#options' => [
        'OVERRIDE' => $this->t('Override - Update existing keys.'),
        'KEEP' => $this->t('Keep - Do not overwrite.'),
      ],
      '#default_value' => $settings['import_options']['force_mode'] ?? 'OVERRIDE',
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::validateConfigurationForm($form, $form_state);

    $values = $form_state->getValue('settings');

    if (isset($values['mapping_wrapper']['namespace_mapping'])) {
      $mappings = $values['mapping_wrapper']['namespace_mapping'];
      $clean_mappings = [];

      foreach ($mappings as $row) {
        if (isset($row['types']) && is_string($row['types'])) {
          $json_decoded = json_decode($row['types'], TRUE);

          if (json_last_error() === JSON_ERROR_NONE && is_array($json_decoded)) {
            $clean_types = [];
            foreach ($json_decoded as $tag) {
              if (isset($tag['value'])) {
                $clean_types[] = $tag['value'];
              }
            }
            $row['types'] = $clean_types;
          }
          else {
            $row['types'] = array_values(array_filter(explode(',', $row['types'])));
          }
        }
        if (!empty($row['namespace']) || !empty($row['types'])) {
          unset($row['remove']);
          $clean_mappings[] = $row;
        }
      }

      $values['namespace_mapping'] = $clean_mappings;
      unset($values['mapping_wrapper']);
      $form_state->setValue('settings', $values);
    }
  }

  /**
   * Submit Handler: Add Row.
   *
   * @param array $form
   *   The configuration form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function addOne(array &$form, FormStateInterface $form_state): void {
    $form_state->setRebuild();
  }

  /**
   * Submit Handler: Remove Row.
   *
   * @param array $form
   *   The configuration form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function removeOne(array &$form, FormStateInterface $form_state): void {
    // 1. Find which button was clicked.
    $trigger = $form_state->getTriggeringElement();
    $parents = $trigger['#parents'];

    // Structure is [settings, mapping_wrapper, namespace_mapping, INDEX, remove]
    // We look for 'namespace_mapping' and take the next key.
    $key_index = array_search('namespace_mapping', $parents);

    if ($key_index !== FALSE && isset($parents[$key_index + 1])) {
      $index_to_remove = $parents[$key_index + 1];

      // 2. Manipulate User Input (Must update input so rebuild sees the change).
      $user_input = $form_state->getUserInput();

      // Find the input array location (tmgmt allows variable nesting).
      $path = [];
      if (isset($user_input['settings']['mapping_wrapper']['namespace_mapping'])) {
        $path = ['settings', 'mapping_wrapper', 'namespace_mapping'];
      }
      elseif (isset($user_input['mapping_wrapper']['namespace_mapping'])) {
        $path = ['mapping_wrapper', 'namespace_mapping'];
      }

      if (!empty($path)) {
        // Get current data.
        $data = NestedArray::getValue($user_input, $path);

        // Remove the specific row.
        if (isset($data[$index_to_remove])) {
          unset($data[$index_to_remove]);
        }

        // Re-index array so it is 0, 1, 2 again.
        $data = array_values($data);

        // Update input and state.
        NestedArray::setValue($user_input, $path, $data);
        $form_state->setUserInput($user_input);
        $form_state->set('tolgee_rows', count($data));
      }
    }

    $form_state->setRebuild();
  }

  /**
   * AJAX Callback: Finds the wrapper dynamically.
   *
   * @param array $form
   *   The configuration form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array|null
   *   The form element to render or NULL.
   */
  public function addmoreCallback(array &$form, FormStateInterface $form_state): ?array {
    $trigger = $form_state->getTriggeringElement();

    $parents = $trigger['#array_parents'];
    $wrapper_index = array_search('mapping_wrapper', $parents);

    if ($wrapper_index !== FALSE) {
      $path = array_slice($parents, 0, $wrapper_index + 1);
      return NestedArray::getValue($form, $path);
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function checkoutSettingsForm(array $form, FormStateInterface $form_state, JobInterface $job) {
    $form['tags'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Additional Tags (Labels)'),
      '#description' => $this->t('Optional: Add comma-separated tags to this export (e.g., "urgent, marketing").'),
    ];
    $translator = $job->getTranslator();
    $translator_plugin = $translator->getPlugin();
    $namespace = $job->getSetting('namespace');
    $default_namespace = $translator->getSetting('namespace');

    if ((empty($namespace) || $namespace === $default_namespace) && $translator_plugin instanceof TolgeeTranslator) {
      $enable_namespaces = $translator->getSetting('enable_namespaces') ?? TRUE;
      if ($enable_namespaces) {
        $detected = $translator_plugin->determineNamespace($job);
        if ($detected) {
          $namespace = $detected;
        }
        elseif (empty($namespace)) {
          $namespace = $default_namespace;
        }
      }
    }
    $form['namespace'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Namespace'),
      '#description' => $this->t('The Tolgee Namespace for this export. You can override the auto-detected value here.'),
      '#default_value' => $namespace,
    ];
    return $form;
  }

}