<?php

namespace Drupal\Tests\tmgmt_tolgee_reverse_sync\Unit;

use Drupal\Core\TypedData\TypedDataInterface;
use Drupal\Core\Field\FieldItemInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\tmgmt_tolgee_reverse_sync\EventSubscriber\TolgeeReverseSyncSubscriber;
use Drupal\core_event_dispatcher\Event\Entity\EntityUpdateEvent;
use Drupal\tmgmt\JobItemInterface;
use Drupal\tmgmt\JobInterface;
use Drupal\tmgmt\TranslatorInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\Query\QueryInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Component\Datetime\TimeInterface;

use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the TolgeeReverseSyncSubscriber.
 */
#[CoversClass(\Drupal\tmgmt_tolgee_reverse_sync\EventSubscriber\TolgeeReverseSyncSubscriber::class)]
#[Group('tmgmt_tolgee')]
class TolgeeReverseSyncSubscriberTest extends UnitTestCase {

  /**
   * The subscriber under test.
   *
   * @var \Drupal\tmgmt_tolgee_reverse_sync\EventSubscriber\TolgeeReverseSyncSubscriber
   */
  protected $subscriber;

  /**
   * The mocked entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * The config factory mock.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $configFactory;

  /**
   * The current user mock.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $currentUser;

  /**
   * The time service mock.
   *
   * @var \Drupal\Component\Datetime\TimeInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $time;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // 1. Mock Container Services
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->currentUser = $this->createMock(AccountProxyInterface::class);

    $container = new ContainerBuilder();
    $container->set('entity_type.manager', $this->entityTypeManager);
    $container->set('config.factory', $this->configFactory);
    $container->set('current_user', $this->currentUser);
    \Drupal::setContainer($container);

    $this->time = $this->createMock(TimeInterface::class);

    $this->subscriber = new TolgeeReverseSyncSubscriber(
      $this->entityTypeManager,
      $this->configFactory,
      $this->currentUser,
      $this->time
    );
  }

  /**
   * Tests onEntityUpdate logic.
   */
  public function testOnEntityUpdate() {
    // 1. Mock Event & Entity
    $event = $this->createMock(EntityUpdateEvent::class);
    $entity = $this->createMock(ContentEntityInterface::class);
    $event->method('getEntity')->willReturn($entity);

    $entity->method('isNew')->willReturn(FALSE);
    $entity->method('getEntityTypeId')->willReturn('node');
    $entity->method('id')->willReturn(123);

    // 2. Mock TMGMT Job Item Query
    $storage = $this->createMock(EntityStorageInterface::class);
    $query = $this->createMock(QueryInterface::class);

    $this->entityTypeManager->method('getStorage')->with('tmgmt_job_item')->willReturn($storage);
    $storage->method('getQuery')->willReturn($query);
    $query->method('condition')->willReturnSelf();
    $query->method('accessCheck')->willReturnSelf();
    // Found one job item.
    $query->method('execute')->willReturn([1 => '1']);

    // 3. Mock Job Item Loading
    $jobItem = $this->createMock(JobItemInterface::class);
    $storage->method('loadMultiple')->willReturn([1 => $jobItem]);

    // 4. Mock Translations Loop
    $langFR = $this->createMock(LanguageInterface::class);
    $entity->method('getTranslationLanguages')->willReturn(['fr' => $langFR]);
    $entity->method('hasTranslation')->with('fr')->willReturn(TRUE);

    $translatedEntity = $this->createMock(ContentEntityInterface::class);
    $entity->method('getTranslation')->with('fr')->willReturn($translatedEntity);

    // CRITICAL: Simulate local change detected.
    $translatedEntity->method('hasTranslationChanges')->willReturn(TRUE);

    // 5. Mock Job & Translator Checks
    $job = $this->createMock(JobInterface::class);
    $jobItem->method('getJob')->willReturn($job);

    $config = $this->createMock(ImmutableConfig::class);
    $config->method('get')->with('enable_reverse_sync')->willReturn(TRUE);
    $this->configFactory->method('get')->willReturn($config);

    $translator = $this->createMock(TranslatorInterface::class);
    $translator->method('getPluginId')->willReturn('tolgee');
    $job->method('getTranslator')->willReturn($translator);
    $job->method('getTargetLangcode')->willReturn('fr');

    // 6. Mock State Check (Must be ACCEPTED to trigger logic)
    $jobItem->method('getState')->willReturn(JobItemInterface::STATE_ACCEPTED);

    // 7. Mock Data & Content comparison
    // Structure says we have 'body'.
    $jobItem->method('getData')->willReturn([
      'body' => [
    // TMGMT structure can be deep.
        'value' => [
          '#text' => 'Old Value',
          '#translation' => ['#text' => 'Old Translation'],
        ],
      ],
    ]);

    // Entity return 'New Translation'.
    $fieldList = $this->createMock(FieldItemListInterface::class);
    $fieldItem = $this->createMock(FieldItemInterface::class);

    // Mock accessing the field chain for getLocalValue.
    $translatedEntity->method('hasField')->willReturn(TRUE);
    $translatedEntity->method('get')->willReturn($fieldList);
    $fieldList->method('offsetExists')->willReturn(TRUE);
    $fieldList->method('get')->willReturn($fieldItem);
    // Mock getting the 'value' property.
    $propertyItem = $this->createMock(TypedDataInterface::class);
    $propertyItem->method('getValue')->willReturn('New Translation');
    $fieldItem->method('get')->with('value')->willReturn($propertyItem);

    // 8. EXPECTATIONS
    // We expect setState(REVIEW) first
    $jobItem->expects($this->once())->method('setState')->with(JobItemInterface::STATE_REVIEW);

    // We expect updateData to be called via forceUpdate logic
    // We do NOT expect addTranslatedData anymore for the force flow.
    $jobItem->expects($this->once())->method('updateData');

    // We expect SAVE.
    $jobItem->expects($this->once())->method('save');

    // Run.
    $this->subscriber->onEntityUpdate($event);
  }

}
