<?php

namespace Drupal\toc_api\Tests;

use Drupal\Core\Render\RenderContext;
use Drupal\KernelTests\KernelTestBase;

/**
 * Tests TOC API manager.
 *
 * @group toc_api
 */
class TocManagerTest extends KernelTestBase {

  /**
   * Modules to enable.
   *
   * @var string[]
   */
  protected static $modules = ['system', 'toc_api'];

  /**
   * The default theme.
   *
   * @var string
   */
  protected $defaultTheme = 'stark';

  /**
   * The content.
   *
   * @var string
   */
  protected string $contentToc;

  /**
   * Empty test to avoid error about there being no tests.
   *
   * Remove when re-enabling disabledTestRender().
   *
   * @return void
   *   No return value.
   */
  public function testRender(): void {
    $this->markTestSkipped('Temporarily skipping until real tests are added.');
  }

  /**
   * Tests TOC rendering. Temporarily disabled.
   *
   * @return mixed
   *   Could return full content (toc tables) or NULL
   */
  public function disabledTestRender() {
    /** @var \Drupal\Core\Render\RendererInterface $renderer */
    $renderer = \Drupal::service('renderer');
    /** @var \Drupal\toc_api\TocManager $toc_manager */
    $toc_manager = \Drupal::service('toc_api.manager');
    /** @var \Drupal\toc_api\TocBuilder $toc_builder */
    $toc_builder = \Drupal::service('toc_api.builder');

    // Create render toc and content functions with context.
    $render_toc = function ($toc) use ($toc_builder, $renderer) {
      return $renderer->executeInRenderContext(new RenderContext(), function () use ($toc_builder, $toc) {
        $contentToc = $toc_builder->renderToc($toc);
        $this->content = $contentToc;
        return $contentToc;
      });
    };
    $render_content = function ($toc) use ($toc_builder, $renderer) {
      return $renderer->executeInRenderContext(new RenderContext(), function () use ($toc_builder, $toc) {
        $contentToc = $toc_builder->renderContent($toc);
        $this->content = $contentToc;
        return $contentToc;
      });
    };

    // Check get and reset Toc.
    $toc = $toc_manager->create('toc_test', '<h2>header 2</h2>', []);
    $this->assertNotNull($toc_manager->getToc('toc_test'));
    $this->assertSame($toc->getIndex(), $toc_manager->getToc('toc_test')->getIndex());
    $toc_manager->reset('toc_test');
    $this->assertNull($toc_manager->getToc('toc_test'));

    // Check default TOC options.
    $toc = $toc_manager->create('toc_test', '<h2>header 2</h2><h3>header 3</h3><h4>header 4</h4><h4>header 4</h4><h2>header 2</h2>', []);
    $render_toc($toc);
    $this->assertNotEmpty($this->cssSelect('.toc-desktop.toc.toc-tree'), 'Toc tree exists');
    $this->assertRaw('<h3>Table of Contents</h3>');
    $this->assertRaw('<a href="#header-2">header 2</a>');
    $this->assertRaw('<a href="#header-3">header 3</a>');
    $this->assertRaw('<a href="#header-4">header 4</a>');
    $this->assertRaw('<a href="#header-4-01">header 4</a>');
    $this->assertRaw('<a href="#header-2-01">header 2</a>');
    $this->assertNotEmpty($this->cssSelect('.toc-mobile.toc.toc-menu'), 'Toc menu exists');
    $this->assertRaw('<option value="">Table of Contents</option>');
    $this->assertRaw('<option value="#header-2">1) header 2</option>');
    $this->assertRaw('<option value="#header-3">1.1) header 3</option>');
    $this->assertRaw('<option value="#header-4">1.1.1) header 4</option>');
    $this->assertRaw('<option value="#header-4-01">1.1.2) header 4</option>');
    $this->assertRaw('<option value="#header-2-01">2) header 2</option>');
    $render_content($toc);
    $this->assertMatchesRegularExpression('|<a href="#top" class="back-to-top">Back to top</a>\s+<h2|s', 'Back to top before h2');
    $this->assertMatchesRegularExpression('|<a href="#top" class="back-to-top">Back to top</a>\s+$|s', 'Back to top at the bottom');
    $this->assertRaw('<a href="#top" class="back-to-top">Back to top</a>');
    $this->assertRaw('<h2 id="header-2"><span>1) </span>header 2</h2>');
    $this->assertRaw('<h3 id="header-3"><span>1.1) </span>header 3</h3>');
    $this->assertRaw('<h4 id="header-4"><span>1.1.1) </span>header 4</h4>');
    $this->assertRaw('<h4 id="header-4-01"><span>1.1.2) </span>header 4</h4>');
    $this->assertRaw('<h2 id="header-2-01"><span>2) </span>header 2</h2>');

    // Check list style type.
    $options = [
      'headers' => [
        'h2' => ['number_type' => 'decimal'],
        'h3' => ['number_type' => 'lower-alpha'],
        'h4' => ['number_type' => 'lower-roman'],
      ],
    ];
    $toc = $toc_manager->create('toc_test', '<h2>header 2</h2><h3>header 3</h3><h4>header 4</h4><h4>header 4</h4><h2>header 2</h2>', $options);
    $render_toc($toc);
    $this->assertMatchesRegularExpression('|<ol class="decimal">.*?<ol class="lower-alpha">.*?<ol class="lower-roman">|s',
      // Contenu à tester.
      $this->content,
      'Failed asserting that the HTML structure matches the expected TOC format.'
    );
    $render_content($toc);
    $this->assertRaw('<h2 id="header-2"><span>1) </span>header 2</h2>');
    $this->assertRaw('<h3 id="header-3"><span>1.a) </span>header 3</h3>');
    $this->assertRaw('<h4 id="header-4"><span>1.a.i) </span>header 4</h4>');
    $this->assertRaw('<h4 id="header-4-01"><span>1.a.ii) </span>header 4</h4>');
    $this->assertRaw('<h2 id="header-2-01"><span>2) </span>header 2</h2>');

    // Check custom attributes and html.
    $toc = $toc_manager->create('toc_test', '<h2 class="custom-class"><b>header</b> 2</h2><h2>header 2</h2>', []);
    $render_toc($toc);
    $this->assertRaw('<a href="#header-2"><b>header</b> 2</a>');
    $this->assertRaw('<option value="#header-2">1) header 2</option>');
    $this->assertRaw('<option value="#header-2-01">2) header 2</option>');
    $render_content($toc);
    $this->assertRaw('<h2 class="custom-class" id="header-2"><span>1) </span><b>header</b> 2</h2>');
    $this->assertRaw('<h2 id="header-2-01"><span>2) </span>header 2</h2>');

    // Check custom back to top.
    $options = [
      'top_label' => 'TOP',
      'top_min' => 3,
      'top_max' => 4,
    ];
    $toc = $toc_manager->create('toc_test', '<h2>header 2</h2><h3>header 3</h3><h4>header 4</h4><h4>header 4</h4><h2>header 2</h2>', $options);
    $render_content($toc);

    $this->assertNoPatternLocal('|<a href="#top" class="back-to-top">TOP</a>\s+<h2|s', 'No back to top before h2');
    $this->assertNoPatternLocal('|<a href="#top" class="back-to-top">TOP</a>\s+$|s', 'No back to top at the bottom');
    $this->assertMatchesRegularExpression('|<a href="#top" class="back-to-top">TOP</a>\s+<h3 id="header-3"><span>1.1\) </span>header 3</h3>|s', 'Back to top before h3');
    $this->assertMatchesRegularExpression('|<a href="#top" class="back-to-top">TOP</a>\s+<h4 id="header-4"><span>1.1.1\) </span>header 4</h4>|s', 'Back to top before first h4');
    $this->assertMatchesRegularExpression('|<a href="#top" class="back-to-top">TOP</a>\s+<h4 id="header-4-01"><span>1.1.2\) </span>header 4</h4>|s', 'Back to top before second h4');

    // Check list style type = 'none' and menu indent.
    $options = [
      'default' => ['number_type' => 'none'],
    ];
    $toc = $toc_manager->create('toc_test', '<h2>header 2</h2><h3>header 3</h3><h4>header 4</h4><h4>header 4</h4><h2>header 2</h2>', $options);
    $render_toc($toc);
    $this->assertRaw('<ol class="none">');
    $render_content($toc);

    // Check unordered list when type = FALSE.
    $options = [
      'number_path' => FALSE,
      'default' => [
        'number_type' => FALSE,
      ],
    ];
    $toc = $toc_manager->create('toc_test', '<h2>header 2</h2><h3>header 3</h3><h4>header 4</h4><h4>header 4</h4><h2>header 2</h2>', $options);
    $render_toc($toc);
    $this->assertNoRaw('<ol');
    $this->assertRaw('<ul');
    $this->assertRaw('<option value="#header-2">header 2</option>');
    $this->assertRaw('<option value="#header-3">-- header 3</option>');
    $this->assertRaw('<option value="#header-4">---- header 4</option>');
    $this->assertRaw('<option value="#header-4-01">---- header 4</option>');
    $render_content($toc);
    $this->assertRaw('<h2 id="header-2">header 2</h2>');
    $this->assertRaw('<h3 id="header-3">header 3</h3>');
    $this->assertRaw('<h4 id="header-4">header 4</h4>');
    $this->assertRaw('<h4 id="header-4-01">header 4</h4>');
  }

  /**
   * Wrapper to handle assertRaw equivalent in PHPUnit.
   */
  protected function assertRaw(mixed $needle, mixed $message = ''): void {
    $this->assertStringContainsString($needle, $this->content, $message);
  }

  /**
   * Wrapper to handle assertNoRaw equivalent in PHPUnit.
   */
  protected function assertNoRaw(mixed $needle, mixed $message = ''): void {
    $this->assertStringNotContainsString($needle, $this->content, $message);
  }

  /**
   * Wrapper to handle assertNoPattern equivalent in PHPUnit.
   */
  protected function assertNoPatternLocal(mixed $pattern, mixed $message = ''): void {
    $this->assertDoesNotMatchRegularExpression($pattern, $this->content, $message);
  }

}
