<?php

declare(strict_types=1);

namespace Drupal\toc_twig_filter\Twig;

use Drupal\Component\Render\MarkupInterface;
use Drupal\toc_api\Entity\TocType;
use Drupal\toc_api\TocBuilderInterface;
use Drupal\toc_api\TocManagerInterface;
use Twig\Extension\AbstractExtension;
use Twig\TwigFilter;

/**
 * Provides a filter to display a table of contents.
 */
class TocTwigExtension extends AbstractExtension {

  /**
   * The TOC manager.
   *
   * @var \Drupal\toc_api\TocManagerInterface
   */
  protected TocManagerInterface $tocManager;

  /**
   * The TOC builder.
   *
   * @var \Drupal\toc_api\TocBuilderInterface
   */
  protected TocBuilderInterface $tocBuilder;

  /**
   * Constructs a new TocTwigExtension.
   *
   * @param \Drupal\toc_api\TocManagerInterface $toc_manager
   *   The TOC manager.
   * @param \Drupal\toc_api\TocBuilderInterface $toc_builder
   *   The TOC builder.
   */
  public function __construct(TocManagerInterface $toc_manager, TocBuilderInterface $toc_builder) {
    $this->tocManager = $toc_manager;
    $this->tocBuilder = $toc_builder;
  }

  /**
   * {@inheritdoc}
   */
  public function getFilters(): array {
    return [
      new TwigFilter('toc', [$this, 'build']),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getName(): string {
    return 'toc';
  }

    /**
   * Constructs a table of contents markup.
   *
   * Examples:
   * {% set body = content.body|render|toc %}
   * {{ body.toc }}
   * {{ body.content }}
   *
   * {% set body = content.body|render|toc('full') %}
   * {{ body.toc }}
   * {{ body.content }}
   *
   * {% set toc_options = {
   *   'header_min': 2,
   *   'header_max': 4,
   * } %}
   * {% set body = content.body|render|toc(toc_options) %}
   * {{ body.toc }}
   * {{ body.content }}
   *
   * @param string|\Drupal\Core\Render\MarkupInterface $render
   *   The render of the field to make table of contents of.
   * @param string|array $options
   *   The TOC API type machine name or a TOC options array.
   *
   * @return array
   *   An associative array containing the table of contents and the content.
   */
  public function build(string|MarkupInterface $render, string|array $options = 'default', ): array {
    if ($render instanceof MarkupInterface) {
      $render = $render->__toString();
    }

    if (is_string($options)) {
      $toc_type = TocType::load($options);
      $options = ($toc_type) ? $toc_type->getOptions() : [];
    }

    $toc = $this->tocManager->create('toc_twig_filter', $render, $options);

    return [
      'toc' => $this->tocBuilder->buildToc($toc),
      'content' => $this->tocBuilder->buildContent($toc),
    ];
  }

}
