<?php

namespace Drupal\toggle_channels\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Database;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class that implements the configuration page.
 */
class SettingsForm extends ConfigFormBase {
  /**
   * Database.
   *
   * @var \Drupal\Core\Database\Database
   */
  protected $database;

  /**
   * Config settings.
   *
   * @var string
   */
  const SETTINGS = 'toggle_channels.settings';

  const FORM_ID = 'toggle_channels_admin_settings';

  /**
   * ConfigForm constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Config factory service.
   * @param \Drupal\Core\Database\Connection $database
   *   Database service to get Connection.
   */
  public function __construct(ConfigFactoryInterface $config_factory, Connection $database) {
    $this->configFactory = $config_factory;
    $this->database = $database;
  }

  /**
   * Lazy building.
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('database')
    );
  }

  /**
   * {@inheritDoc}
   */
  public function getFormId() {
    return self::FORM_ID;
  }

  /**
   * {@inheritDoc}
   */
  protected function getEditableConfigNames() {
    return [
      self::SETTINGS,
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Get configurations.
    $config = $this->configFactory()->getEditable(self::SETTINGS);

    $form['logger'] = [
      '#type' => 'details',
      '#open' => TRUE,
      '#title' => $this->t('logger.factory'),
    ];

    $form['logger']['toggle_logger'] = [
      '#type' => 'checkbox',
      '#title' => $this->t("Disable logger.factory service's channels"),
      '#description' => $this->t('Enabling this option all below checked channels will be disabled and no more logged.'),
      '#default_value' => $config->get('toggle_logger') ?? FALSE,
    ];

    /** @var array $channels */
    $channels = $this->getLogChannels();

    /** @var array $options */
    $options = [];

    foreach ($channels as $channel) {
      $options[$channel->type] = $channel->type;
    }

    $form['logger']['toggle_channels'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Select channels to disable'),
      '#default_value' => $config->get('toggle_channels') ?? [],
      '#options' => $options,
    ];

    // @todo Change the autogenerated stub.
    return parent::buildForm($form, $form_state);
  }

  /**
   * Submit form.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Get configurations.
    $config = $this->configFactory()->getEditable(self::SETTINGS);

    $config
      ->set('toggle_logger', $form_state->getValue('toggle_logger'))
      ->set('toggle_channels', $form_state->getValue('toggle_channels'))
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Return all watchdog channels.
   *
   * @return mixed
   *   Return value.
   */
  private function getLogChannels() {
    $query = $this->database->select('watchdog', 'w');

    return $query->fields('w', ['type'])->distinct(TRUE)
      ->execute()->fetchAll();
  }

}
