<?php

namespace Drupal\token_aware_views_field\Plugin\views\field;

use Drupal\Core\Form\FormStateInterface;
use Drupal\metatag\MetatagToken;
use Drupal\token\Token;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides token aware field handler.
 *
 * @ViewsField("token_aware_views_field")
 */
class TokenAwareViewsField extends FieldPluginBase {

  /**
   * Constructs a new TokenAwareViewsField instance.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\metatag\MetatagToken|Token $token
   *   The metatag.token or the token service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, protected MetatagToken|Token $token) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->has('metatag.token') ? $container->get('metatag.token') : $container->get('token')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['token'] = ['default' => ''];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);
    $form['token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Token'),
      '#default_value' => $this->options['token'],
      '#description' => $this->t('Enter a token like [node:meta_tags:title].'),
    ];
    if ($this->token instanceof MetatagToken) {
      $form['metatag_browser'] = $this->token->tokenBrowser([$this->configuration['entity_type']]);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values) {
    if (!$entity = $this->getEntity($values)) {
      return '';
    }
    // To work around https://www.drupal.org/project/metatag/issues/3483973
    // pass the entity in options too.
    $data = [$entity->getEntityTypeId() => $entity];
    $options = ['entity' => $entity];
    $replacement = $this->token->replace($this->options['token'], $data, $options);
    if (str_contains($replacement, '[')) {
      $replacement = $this->token->replace($replacement, $data, $options);
    }
    return $replacement;
  }

  public function query() {
    // Nothing to do, this class only needs the entity.
  }

}
