<?php

use Drupal\Core\Render\BubbleableMetadata;

/**
 * Implements hook_tokens().
 */
function token_aware_views_field_tokens($type, $tokens, array $data, array $options, BubbleableMetadata $bubbleable_metadata) {
  $replacements = [];
  if (str_starts_with($type, 'token_aware_views_field')) {
    $field_name = str_replace('token_aware_views_field', '', $type);
    $entity = $data[$options['token_aware_views_field_type']];
    $source_data = $entity->get($field_name)->value;
    if (isset($options['token_aware_views_field_callback'])) {
      $source_data = ($options['token_aware_views_field_callback'])($source_data);
    }
    foreach ($tokens as $token => $raw) {
      if (isset($source_data[$token])) {
        $replacements[$raw] = $source_data[$token];
      }
    }
  }
  return $replacements;
}

/**
 * Replace sub tokens with data from a serialized field.
 *
 * For example, if you have a JSON encoded field called "json_data" and it
 * contains {"foo":"bar"} then this helper will help with [node:json_data:foo].
 * Set $field_name to json_data and $decode_callable to 'json_decode'.
 *
 * @param string $field_name
 *   An entity field name.
 * @param $type
 *   The type of token being replaced but also an entity type.
 * @param $tokens
 *   The tokens.
 * @param array $data
 *   This array must contain an entity keyed by $type.
 * @param array $options
 *   See Token::generate().
 * @param \Drupal\Core\Render\BubbleableMetadata $bubbleable_metadata
 *   See Token::generate().
 * @param callable|null $decode_callable
 *   This module is mostly used with fields storing serialized data. This is
 *   the decode callback for the serialization. This is optional because a
 *   field might be handling decoding by itself.
 *
 * @return array
 */
function token_aware_views_field_token_helper(string $field_name, $type, $tokens, array $data, array $options, BubbleableMetadata $bubbleable_metadata, ?callable $decode_callable = NULL) {
  $replacements = [];
  $token_service = \Drupal::token();
  $my_type = "token_aware_views_field$field_name";
  if ($prefixed_tokens = $token_service->findWithPrefix($tokens, $field_name)) {
    $options['token_aware_views_field_type'] = $type;
    $options['token_aware_views_field_callback'] = $decode_callable;
    $replacements += $token_service->generate($my_type, $prefixed_tokens, $data, $options + ['token_aware_views_field_type' => $type], $bubbleable_metadata);
  }

  return $replacements;
}