(function (Drupal, drupalSettings) {
  /* global tinyMCE, CKEDITOR, CodeMirror */
  /**
   * Inserts text at the cursor position in a textarea.
   *
   * @param {HTMLTextAreaElement} editor
   *   The textarea element.
   * @param {string} content
   *   The content to insert.
   */
  function insertAtCursor(editor, content) {
    // Record the current scroll position.
    const scroll = editor.scrollTop;

    // IE support.
    if (document.selection) {
      editor.focus();
      const sel = document.selection.createRange();
      sel.text = content;
    }

    // Mozilla/Firefox/Netscape 7+ support.
    else if (editor.selectionStart !== null) {
      const startPos = editor.selectionStart;
      const endPos = editor.selectionEnd;
      const valueBefore = editor.value.substring(0, startPos);
      const valueAfter = editor.value.substring(endPos);
      editor.value = `${valueBefore}${content}${valueAfter}`;
      const newCursorPos = startPos + content.length;
      editor.selectionStart = newCursorPos;
      editor.selectionEnd = newCursorPos;
    }

    // Fallback, just add to the end of the content.
    else {
      editor.value += content;
    }

    // Ensure the textarea does not unexpectedly scroll.
    editor.scrollTop = scroll;
    editor.focus();
  }

  Drupal.behaviors.tokenTree = {
    attach(context, settings) {
      once(
        'htmx-after-swap-listener',
        'table.token-tree tbody',
        context,
      ).forEach((tbody) => {
        tbody.addEventListener('htmx:afterSwap', (e) => {
          const button = e.target.querySelector(
            `button[data-tt-target="${e.detail.target.dataset.ttId}"]`,
          );
          if (button) {
            button.focus();
          }
        });
      });
      once('token-tree', '.token-toggle', context).forEach((toggle) => {
        const handleToggle = () => {
          const thisRow = toggle.closest('tr');
          const tbody = thisRow.closest('tbody');
          const parentId = toggle.dataset.parent;
          const isExpanding = toggle.getAttribute('aria-expanded') !== 'true';

          toggle.setAttribute('aria-expanded', isExpanding ? 'true' : 'false');
          thisRow.classList.toggle('expanded', isExpanding);
          thisRow.classList.toggle('collapsed', !isExpanding);

          let selectorId = parentId;
          if (toggle.hasAttribute('data-tt-target')) {
            selectorId = toggle.getAttribute('data-tt-target');
          }
          const directSelector = `[data-tt-parent-id="${selectorId}"]`;
          const directChildren = tbody.querySelectorAll(directSelector);

          directChildren.forEach((child) => {
            child.classList.toggle('hidden');
            if (isExpanding) {
              const childToggle = child.querySelector('.token-branch-toggle');
              if (childToggle) {
                childToggle.setAttribute('aria-expanded', 'false');
                child.classList.remove('expanded');
                child.classList.add('collapsed');

                let selectorId = childToggle.dataset.parent;
                if (childToggle.hasAttribute('data-tt-target')) {
                  selectorId = childToggle.getAttribute('data-tt-target');
                }
                const descendantSelector = `[data-tt-parent-id="${selectorId}"]`;
                tbody.querySelectorAll(descendantSelector).forEach((desc) => {
                  desc.classList.add('hidden');
                });
              }
            }
          });

          // Collapse: hide all descendants.
          if (!isExpanding) {
            let selectorId = parentId;
            if (toggle.hasAttribute('data-tt-target')) {
              selectorId = toggle.getAttribute('data-tt-target');
            }
            const selector = [
              `[data-tt-parent-id="${selectorId}"]`,
              `[data-tt-parent-id^="${selectorId}--"]`,
            ].join(', ');

            tbody.querySelectorAll(selector).forEach((desc) => {
              desc.classList.add('hidden');
              const descToggle = desc.querySelector('.token-branch-toggle');
              if (descToggle) {
                descToggle.setAttribute('aria-expanded', 'false');
                desc.classList.remove('expanded');
                desc.classList.add('collapsed');
              }
            });
          }
        };

        // CLICK: Triggered by mouse OR Enter key.
        toggle.addEventListener('click', (e) => {
          e.stopPropagation();
          handleToggle();
        });

        // KEYDOWN: Only handle Space.
        toggle.addEventListener('keydown', (e) => {
          if (e.key === ' ') {
            e.preventDefault();
            e.stopPropagation();
            handleToggle();
          }
        });
      });
    },
  };

  Drupal.behaviors.tokenInsert = {
    attach(context, settings) {
      // Keep track of which textfield was last selected/focused.
      const textFields = context.querySelectorAll(
        'textarea, input[type="text"]',
      );
      textFields.forEach((field) => {
        field.addEventListener('focus', () => {
          drupalSettings.tokenFocusedField = field;
        });
      });

      if (Drupal.CKEditor5Instances) {
        Drupal.CKEditor5Instances.forEach((editor) => {
          editor.editing.view.document.on(
            'change:isFocused',
            (event, data, isFocused) => {
              if (isFocused) {
                drupalSettings.tokenFocusedCkeditor5 = editor;
              }
            },
          );
        });
      }

      once(
        'token-click-insert',
        '.token-click-insert .token-key',
        context,
      ).forEach((token) => {
        // Build the <a> element that will replace the token
        const link = document.createElement('a');
        link.href = '#';
        link.title = Drupal.t('Insert this token into your form');
        link.textContent = token.textContent;
        link.setAttribute('role', 'button');
        link.setAttribute('tabindex', '0');
        link.addEventListener('click', (e) => {
          e.preventDefault();
          const content = link.textContent;

          // Always work in normal text areas that currently have focus.
          if (
            drupalSettings.tokenFocusedField &&
            (drupalSettings.tokenFocusedField.tokenDialogFocus ||
              drupalSettings.tokenFocusedField.tokenHasFocus)
          ) {
            insertAtCursor(drupalSettings.tokenFocusedField, content);
          }
          // Direct tinyMCE support.
          else if (typeof tinyMCE !== 'undefined' && tinyMCE.activeEditor) {
            tinyMCE.activeEditor.execCommand(
              'mceInsertContent',
              false,
              content,
            );
          }
          // Direct CKEditor support. Only works if the field currently has focus,
          // which is unusual since the dialog is open.
          else if (
            typeof CKEDITOR !== 'undefined' &&
            CKEDITOR.currentInstance
          ) {
            CKEDITOR.currentInstance.insertHtml(content);
          }
          // Direct CodeMirror support.
          else if (
            typeof CodeMirror !== 'undefined' &&
            drupalSettings.tokenFocusedField
          ) {
            const cmWrapper =
              drupalSettings.tokenFocusedField.closest('.CodeMirror');
            if (cmWrapper) {
              const cm = cmWrapper.CodeMirror;
              cm.replaceSelection(content);
              cm.focus();
            }
          }
          // WYSIWYG support, should work in all editors if available.
          else if (Drupal.wysiwyg && Drupal.wysiwyg.activeId) {
            Drupal.wysiwyg.instances[Drupal.wysiwyg.activeId].insert(content);
          }
          // CKeditor module support.
          else if (
            typeof CKEDITOR !== 'undefined' &&
            typeof Drupal.ckeditorActiveId !== 'undefined'
          ) {
            CKEDITOR.instances[Drupal.ckeditorActiveId].insertHtml(content);
          } else if (drupalSettings.tokenFocusedField) {
            insertAtCursor(drupalSettings.tokenFocusedField, content);
          } else if (drupalSettings.tokenFocusedCkeditor5) {
            const editor = drupalSettings.tokenFocusedCkeditor5;
            editor.model.change((writer) => {
              writer.insertText(
                content,
                editor.model.document.selection.getFirstPosition(),
              );
            });
          } else {
            alert(
              Drupal.t('First click a text field to insert your tokens into.'),
            );
          }
        });
        // Keyboard support (Enter / Space).
        link.addEventListener('keydown', (e) => {
          if (e.key === 'Enter' || e.key === ' ') {
            e.preventDefault();
            link.click();
          }
        });
        token.replaceChild(link, token.firstChild);
      });
    },
  };
})(Drupal, drupalSettings);
