<?php

namespace Drupal\token_browser\Element;

use Drupal\Core\Htmx\Htmx;
use Drupal\Core\Render\Attribute\RenderElement;
use Drupal\Core\Render\Element\Table;
use Drupal\Core\Url;
use Drupal\token_browser\Trait\TreeTableTrait;

/**
 * Provides a render element for a token tree table.
 */
#[RenderElement('token_browser_tree_table')]
class TokenBrowserTreeTable extends Table {

  use TreeTableTrait;

  /**
   * {@inheritdoc}
   */
  public function getInfo(): array {
    $class = get_class($this);
    return [
      '#header' => [],
      '#rows' => [],
      '#token_tree' => [],
      '#columns' => ['name', 'token', 'description'],
      '#empty' => '',
      '#show_restricted' => FALSE,
      '#show_nested' => FALSE,
      '#skip_empty_values' => FALSE,
      '#click_insert' => TRUE,
      '#sticky' => FALSE,
      '#responsive' => TRUE,
      '#input' => FALSE,
      '#pre_render' => [
        [$class, 'preRenderTokenTree'],
        [$class, 'preRenderTable'],
      ],
      '#theme' => 'table__token_browser',
      '#attached' => [
        'library' => [
          'token_browser/token_browser',
        ],
      ],
    ];
  }

  /**
   * Pre-render the token tree to transform rows in the token tree.
   *
   * @param array $element
   *   The element to process.
   *
   * @return array
   *   The processed element.
   */
  public static function preRenderTokenTree(array $element): array {
    $multiple_token_types = count($element['#token_tree']) > 1;
    foreach ($element['#token_tree'] as $token_type => $type_info) {
      // Do not show nested tokens.
      if (!empty($type_info['nested']) && empty($element['#show_nested'])) {
        continue;
      }

      if ($multiple_token_types) {
        $row = static::formatRow($token_type, $type_info, $element['#columns'], TRUE);
        $element['#rows'][] = $row;
      }

      foreach ($type_info['tokens'] as $token => $token_info) {
        if (!empty($token_info['restricted']) && empty($element['#show_restricted'])) {
          continue;
        }
        if ($element['#skip_empty_values'] && empty($token_info['value']) && !empty($token_info['parent'])) {
          continue;
        }
        if ($multiple_token_types && !isset($token_info['parent'])) {
          $token_info['parent'] = $token_type;
        }
        $row = static::formatRow($token, $token_info, $element['#columns']);
        $element['#rows'][] = $row;
      }
    }

    // Fill headers if one is not specified.
    if (empty($element['#header'])) {
      $column_map = [
        'name' => t('Name'),
        'token' => t('Token'),
        'value' => t('Value'),
        'description' => t('Description'),
      ];
      foreach ($element['#columns'] as $col) {
        $element['#header'][] = $column_map[$col];
      }
    }

    $element['#attributes']['class'][] = 'token-tree';

    if ($element['#click_insert']) {
      $element['#caption'] = t("Click a token to insert it into the field you've last clicked.");
      $element['#attributes']['class'][] = 'token-click-insert';
    }

    return $element;
  }

  /**
   * Format a row for the token tree table.
   *
   * @param string $token
   *   The token.
   * @param array $token_info
   *   The token info.
   * @param array $columns
   *   The columns to show.
   * @param bool $is_group
   *   (optional) Defaults to FALSE. Whether the row is a token type or a token.
   *
   * @return array
   *   The formatted row.
   */
  protected static function formatRow(string $token, array $token_info, array $columns, bool $is_group = FALSE): array {
    $row = [
      'id' => static::cleanCssIdentifier($token),
      'data-tt-id' => static::cleanCssIdentifier($token),
      'class' => [],
      'data' => [],
    ];

    foreach ($columns as $col) {
      switch ($col) {
        case 'name':
          $wrapper = [
            'data' => [
              '#type' => 'container',
            ],
          ];
          if ($is_group) {
            $wrapper['data'] += static::createToggleButton($token_info['name'], $token, TRUE);
          }
          else {
            if (!empty($token_info['parent']) && isset($token_info['children'])) {
              $wrapper['data'] += static::createToggleButton($token_info['name'], $token, FALSE);
              if (empty($token_info['children'])) {
                $htmx = new Htmx();
                $htmx->get(Url::fromRoute('token_browser.htmx', [
                  'token_type' => $token_info['parent'],
                  'parent' => $token_info['parent'],
                  'token' => $token_info['raw token'],
                ]))
                  ->trigger('click')
                  ->swap('outerHTML')
                  ->target('closest tr');
                $htmx_attributes = $htmx->getAttributes()->toArray();
                $wrapper['data']['button']['#attributes'] += $htmx_attributes;
              }
            }
            else {
              $wrapper['data']['#markup'] = $token_info['name'];
            }
            $wrapper['data']['#attributes']['class'][] = static::getIndentClass($token);
          }
          $row['data'][$col] = $wrapper;
          break;

        case 'token':
          $row['data'][$col]['data'] = $token;
          $row['data'][$col]['class'][] = 'token-key';
          break;

        case 'description':
          $row['data'][$col] = $token_info['description'] ?? '';
          break;

        case 'value':
          $row['data'][$col] = !$is_group && isset($token_info['value']) ? $token_info['value'] : '';
          break;
      }
    }

    if ($is_group) {
      // This is a token type/group.
      $row['class'] = [
        'token-group',
        'collapsed',
      ];
    }
    elseif (!empty($token_info['parent'])) {
      $row['data-tt-parent-id'] = static::cleanCssIdentifier($token_info['parent']);
      $row['class'][] = 'hidden';
      if (isset($token_info['children'])) {
        $row['data-hx-select'] = 'tr';
        $row['class'][] = 'collapsed';
      }
    }

    return $row;
  }

}
