<?php

namespace Drupal\token_browser\Trait;

use Drupal\Component\Utility\Html;

/**
 * Trait for various token tree table methods.
 */
trait TreeTableTrait {

  /**
   * CSS filter for token tree table.
   *
   * @var string[]
   */
  protected static array $cssFilter = [
    ' ' => '-',
    '_' => '-',
    '/' => '-',
    '[' => '-',
    ']' => '',
    ':' => '--',
    '?' => '',
    '<' => '-',
    '>' => '-',
  ];

  /**
   * Get the indent class for a token.
   *
   * @param string $token
   *   The token.
   *
   * @return string
   *   The indent class.
   */
  protected static function getIndentClass(string $token): string {
    $indent = str_replace(['[', ']', ':?'], '', $token);
    $parts = count(explode(':', $indent));

    return 'indent-' . $parts;
  }

  /**
   * Clean a CSS identifier.
   *
   * @param string $id
   *   The CSS identifier to clean.
   *
   * @return string
   *   The cleaned CSS identifier.
   */
  protected static function cleanCssIdentifier(string $id): string {
    return 'token-' . Html::cleanCssIdentifier(trim($id, '[]'), static::$cssFilter);
  }

  /**
   * Helper function to create a toggle button.
   *
   * @param string $value
   *   The button value property.
   * @param string $parent
   *   The data-parent attribute.
   * @param bool $is_group
   *   An indicator to determine button class as group or branch.
   * @param bool $is_expanded
   *   An indicator to determine the aria-expanded attribute.
   *
   * @return array[]
   *   The rendered button.
   */
  protected static function createToggleButton(string $value, string $parent, bool $is_group, bool $is_expanded = FALSE): array {
    return [
      'button' => [
        '#type' => 'html_tag',
        '#tag' => 'button',
        '#value' => $value,
        '#attributes' => [
          'class' => [
            'token-toggle',
            $is_group ? 'token-group-toggle' : 'token-branch-toggle',
          ],
          'type' => 'button',
          'aria-expanded' => $is_expanded ? 'true' : 'false',
          'data-parent' => static::cleanCssIdentifier($parent),
        ],
      ],
    ];
  }

}
