<?php

namespace Drupal\token_browser\Controller;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Htmx\Htmx;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Url;
use Drupal\token_browser\TokenBrowserTreeBuilderInterface;
use Drupal\token_browser\Trait\TreeTableTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Returns tree responses for tokens.
 */
class TokenBrowserController extends ControllerBase {

  use TreeTableTrait;

  public function __construct(
    protected TokenBrowserTreeBuilderInterface $treeBuilder,
    protected RendererInterface $renderer,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('token_browser.tree_builder'),
      $container->get('renderer'),
    );
  }

  /**
   * Page callback to output a token tree as an empty page.
   */
  public function outputTree(Request $request): array {
    $options = $request->query->has('options') ? Json::decode($request->query->get('options')) : [];

    // The option token_types may only be an array OR 'all'. If it is not set,
    // we assume that only global token types are requested.
    $token_types = !empty($options['token_types']) ? $options['token_types'] : [];
    if ($token_types == 'all') {
      $build = $this->treeBuilder->buildAllRenderable($options);
    }
    else {
      $build = $this->treeBuilder->buildRenderable($token_types, $options);
    }

    $build['#cache']['contexts'][] = 'url.query_args:options';
    $build['#title'] = $this->t('Available tokens');

    // If this is an AJAX/modal request, add a wrapping div to the contents so
    // that Drupal.behaviors.tokenTree and Drupal.behaviors.tokenAttach can
    // still find the elements they need to.
    // @see https://www.drupal.org/project/token/issues/2994671
    // @see https://www.drupal.org/node/2940704
    // @see http://danielnouri.org/notes/2011/03/14/a-jquery-find-that-also-finds-the-root-element/
    if ($request->isXmlHttpRequest()) {
      $build['#prefix'] = '<div>';
      $build['#suffix'] = '</div>';
    }

    return $build;
  }

  /**
   * Page callback to output table rows for a token type.
   *
   * @param string $token_type
   *   The token type.
   * @param string $parent
   *   The parent token.
   * @param string $token
   *   The token.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The HTML response with the rows.
   */
  public function outputHtmxRows(string $token_type, string $parent, string $token): Response {
    $token_parts = explode(':', str_replace(['[', ']'], '', $token));
    // The count of token parts tells us how deep to request.
    $recursion = count($token_parts);
    $element = $this->treeBuilder->buildRenderable([$token_type], [
      'global_types' => FALSE,
      'recursion_limit' => $recursion,
    ]);
    $parent_token = current($token_parts);
    $tokens = $element['#token_tree'][$parent_token]['tokens'] ?? [];
    $parent_token_info = $tokens[$token] ?? $element['#token_tree'][$token_type] ?? [];
    $rows = [];
    if (!empty($parent_token_info)) {
      $parent_token_info['parent'] = $parent;
      $parent_token_info['parent_top'] = $parent_token;
      $parent_row = static::formatRow($token, $parent_token_info, $element['#columns'], FALSE, TRUE);
      $rows[] = $parent_row;
      if (isset($parent_token_info['children'])) {
        foreach ($parent_token_info['children'] as $child_token => $child_token_info) {
          $child_token_info['parent_top'] = $parent_token;
          $rows[] = static::formatRow($child_token, $child_token_info, $element['#columns']);
        }
      }
    }

    $build = [
      '#theme' => 'table',
      '#header' => [],
      '#rows' => $rows,
    ];

    $html = $this->renderer->renderRoot($build);

    return new Response($html);
  }

  /**
   * Format a row for the token tree table.
   *
   * @param string $token
   *   The token.
   * @param array $token_info
   *   The token info.
   * @param array $columns
   *   The columns to show.
   * @param bool $is_group
   *   (optional) Defaults to FALSE. Whether the row is a token type or a token.
   * @param bool $is_branch
   *   (optional) Defaults to FALSE. Whether the row is a branch.
   *
   * @return array
   *   The formatted row.
   */
  protected static function formatRow(string $token, array $token_info, array $columns, bool $is_group = FALSE, bool $is_branch = FALSE): array {
    $row = [
      'id' => static::cleanCssIdentifier($token),
      'data-tt-id' => static::cleanCssIdentifier($token),
      'class' => [],
      'data' => [],
    ];
    if ($is_group) {
      // This is a token type/group.
      $row['class'] = [
        'token-group',
        'collapsed',
      ];
    }
    elseif (!empty($token_info['parent'])) {
      $row['data-tt-parent-id'] = static::cleanCssIdentifier($token_info['parent']);
      if (isset($token_info['children'])) {
        $row['class'][] = $is_branch ? 'expanded' : 'collapsed';
      }
    }

    foreach ($columns as $col) {
      switch ($col) {
        case 'name':
          $wrapper = [
            'data' => [
              '#type' => 'container',
            ],
          ];
          if ($is_group) {
            $wrapper['data'] += static::createToggleButton($token_info['name'], $token_info['type'], TRUE);
          }
          else {
            if (!empty($token_info['parent']) && isset($token_info['children'])) {
              $wrapper['data'] += static::createToggleButton($token_info['name'], $token_info['parent'], FALSE, $is_branch);
              $wrapper['data']['button']['#attributes']['data-tt-target'] = $row['data-tt-id'];
              if (empty($token_info['children'])) {
                $htmx = new Htmx();
                $htmx->get(Url::fromRoute('token_browser.htmx', [
                  'token_type' => $token_info['parent_top'],
                  'parent' => $token_info['parent'],
                  'token' => $token,
                ]))
                  ->trigger('click')
                  ->swap('outerHTML')
                  ->target('closest tr');
                $htmx->applyTo($wrapper['data']['button']);
                $row['data-hx-select'] = 'tr';
              }
            }
            else {
              $wrapper['data']['#markup'] = $token_info['name'];
            }
            $wrapper['data']['#attributes']['class'][] = static::getIndentClass($token);
          }
          $row['data'][$col] = $wrapper;
          break;

        case 'token':
          $row['data'][$col]['data'] = $token;
          $row['data'][$col]['class'][] = 'token-key';
          break;

        case 'description':
          $row['data'][$col] = $token_info['description'] ?? '';
          break;

        case 'value':
          $row['data'][$col] = !$is_group && isset($token_info['value']) ? $token_info['value'] : '';
          break;
      }
    }

    return $row;
  }

}
