<?php

declare(strict_types=1);

namespace Drupal\token_browser\Hook;

use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Utility\Token;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\token_browser\TokenBrowserTreeBuilderInterface;

/**
 * Hook implementations for token_browser.
 */
final class TokenBrowserHooks {

  use StringTranslationTrait;

  public function __construct(
    protected readonly Token $token,
    protected readonly TokenBrowserTreeBuilderInterface $treeBuilder,
    protected readonly RendererInterface $renderer,
  ) {
  }

  /**
   * Implements hook_help().
   */
  #[Hook('help')]
  public function help($route_name, RouteMatchInterface $route_match): ?array {
    if ($route_name == 'help.page.token_browser') {
      $token_tree = $this->treeBuilder->buildAllRenderable([
        'click_insert' => FALSE,
        'show_restricted' => TRUE,
        'show_nested' => FALSE,
        'recursion_limit' => 1,
      ]);

      $output = '<h3>' . $this->t('About') . '</h3>';
      $output .= '<p>' . $this->t('The <a href=":project">Token browser</a> module provides a user interface for the site token system. Tokens are specially formatted chunks of text that serve as placeholders for a dynamically generated value. For more information, covering both the token system and the additional tools provided by the Token module, see the <a href=":online">online documentation</a>.', [
        ':online' => 'https://www.drupal.org/documentation/modules/token',
        ':project' => 'https://www.drupal.org/project/token',
      ]) . '</p>';
      $output .= '<h3>' . $this->t('Uses') . '</h3>';
      $output .= '<p>' . $this->t('Your website uses a shared token system for exposing and using placeholder tokens and their appropriate replacement values. This allows for any module to provide placeholder tokens for strings without having to reinvent the wheel. It also ensures consistency in the syntax used for tokens, making the system as a whole easier for end users to use.') . '</p>';
      $output .= '<p>' . $this->t('The list of the currently available tokens on this site are shown below.') . '</p>';

      return [
        '#type' => 'container',
        '#markup' => $output,
        'table' => $token_tree,
      ];
    }

    return NULL;
  }

}
