<?php

declare(strict_types=1);

namespace Drupal\tool_content\Plugin\tool\Tool\Deriver;

use Drupal\Component\Plugin\Derivative\DeriverBase;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Plugin\Discovery\ContainerDeriverInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\TypedData\InputDefinition;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Deriver that creates entity create tools for each fieldable entity type.
 */
class EntityCreateDeriver extends DeriverBase implements ContainerDeriverInterface {

  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EntityTypeBundleInfoInterface $entityTypeBundleInfo,
    protected EntityFieldManagerInterface $entityFieldManager,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, $base_plugin_id) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('entity_field.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getDerivativeDefinitions($base_plugin_definition) {
    if (empty($this->derivatives)) {
      $entity_types = $this->entityTypeManager->getDefinitions();

      foreach ($entity_types as $entity_type_id => $entity_type) {
        // Only process fieldable content entity types
        if (!$entity_type->entityClassImplements(ContentEntityInterface::class)) {
          continue;
        }

        $derivative_id = $entity_type_id;

        // Get base field definitions (not bundle-specific fields)
        $base_field_definitions = $this->entityFieldManager->getBaseFieldDefinitions($entity_type_id);
        $input_definitions = [];

        if($entity_type->getBundleEntityType()) {
          $bundles = $this->entityTypeBundleInfo->getBundleInfo($entity_type_id);
          if (empty($bundles)) {
            continue;
          }
          $input_definitions[$entity_type->getKey('bundle')] = new InputDefinition(
            data_type: 'string',
            label: new TranslatableMarkup('Bundle'),
            required: TRUE,
            description: new TranslatableMarkup('The bundle of the entity to create.'),
            constraints: [
              'Choice' => [
                'choices' => array_keys($bundles),
             ],
            ],
          );
        }
        else {
          $input_definitions[$entity_type->getKey('bundle')] = new InputDefinition(
            data_type: 'string',
            label: new TranslatableMarkup('Bundle'),
            required: TRUE,
            description: new TranslatableMarkup('The bundle of the entity to create.'),
            default_value: $entity_type->id(),
            locked: TRUE,// Default to the entity type ID if no bundles exist
          );
        }

        foreach ($base_field_definitions as $field_name => $field_definition) {
          // Skip computed fields, read-only fields, and auto-generated fields
          if ($field_definition->isComputed() || $field_definition->isReadOnly() || !$field_definition->isDisplayConfigurable('form')) {
            continue;
          }
          $key = $entity_type->hasKey($field_name) ? $entity_type->getKey($field_name) : $field_name;

          // Include configurable base properties like title, status, bundle, etc.
          $input_definitions[$key] = new InputDefinition(
            data_type: $this->getFieldDataType($field_definition),
            label: $field_definition->getLabel(),
            required: $field_definition->isRequired(),
            multiple: $field_definition->getFieldStorageDefinition()->getCardinality() !== 1,
            description: $field_definition->getDescription() ?: new TranslatableMarkup('Property: @name', ['@name' => $key])
          );
          if ($field_definition->getDefaultValueLiteral()) {
            // @todo: what to do here.
            $value = $field_definition->getDefaultValueLiteral();
            if (isset($value[0]['value'])) {
              // For string fields, use the value directly.
              $input_definitions[$key]->setDefaultValue($value[0]['value']);
            }
          }
        }

        $this->derivatives[$derivative_id] = clone $base_plugin_definition;
        $this->derivatives[$derivative_id]->set('id', $base_plugin_definition->id() . ':' . $derivative_id);
        $this->derivatives[$derivative_id]->setLabel(new TranslatableMarkup('Create @entity_type', [
          '@entity_type' => $entity_type->getLabel()
        ]));
        $this->derivatives[$derivative_id]->setDescription(new TranslatableMarkup('Create a new @entity_type with base properties.', [
          '@entity_type' => $entity_type->getLabel()
        ]));
        foreach ($input_definitions as $name => $input_definition) {
          $this->derivatives[$derivative_id]->addInputDefinition($name, $input_definition);
        }
      }
    }

    return $this->derivatives;
  }

  /**
   * Map field types to appropriate data types.
   */
  private function getFieldDataType(FieldDefinitionInterface $field_definition): string {
    $field_type = $field_definition->getType();

    $type_mapping = [
      'string' => 'string',
      'string_long' => 'string',
      'text' => 'string',
      'text_long' => 'string',
      'text_with_summary' => 'string',
      'integer' => 'integer',
      'decimal' => 'float',
      'float' => 'float',
      'boolean' => 'boolean',
      'email' => 'email',
      'link' => 'uri',
      'file' => 'integer', // File ID
      'image' => 'integer', // File ID
      'entity_reference' => 'integer', // Entity ID
      'datetime' => 'datetime_iso8601',
      'timestamp' => 'timestamp',
      'list_string' => 'string',
      'list_integer' => 'integer',
      'list_float' => 'float',
    ];

    return $type_mapping[$field_type] ?? 'string';
  }

}
