<?php

declare(strict_types=1);

namespace Drupal\tool_system\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\system\SystemManager;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ToolBase;
use Drupal\tool\TypedInputs;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the system status tool.
 */
#[Tool(
  id: 'system_status',
  label: new TranslatableMarkup('System Status'),
  description: new TranslatableMarkup('Get the Drupal system status information including warnings and errors.'),
  input_definitions: [],
  output_definitions: [
    'status_report' => new ContextDefinition(
      data_type: 'any',
      label: new TranslatableMarkup("Status Report"),
      description: new TranslatableMarkup("Array containing the system status report data.")
    ),
    'requirements' => new ContextDefinition(
      data_type: 'any',
      label: new TranslatableMarkup("Requirements"),
      description: new TranslatableMarkup("Array of system requirements with their status.")
    ),
    'has_warnings' => new ContextDefinition(
      data_type: 'boolean',
      label: new TranslatableMarkup("Has Warnings"),
      description: new TranslatableMarkup("Whether the system has warnings.")
    ),
    'has_errors' => new ContextDefinition(
      data_type: 'boolean',
      label: new TranslatableMarkup("Has Errors"),
      description: new TranslatableMarkup("Whether the system has errors.")
    ),
  ],
)]
final class SystemStatus extends ToolBase {

  public function __construct(
    array $configuration,
          $plugin_id,
          $plugin_definition,
    protected TypedInputs $typedInputs,
    protected SystemManager $systemManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $typedInputs);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('typed.inputs'),
      $container->get('system.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    try {
      // Get the system requirements
      $requirements = $this->systemManager->listRequirements();

      // Process the requirements to extract status information
      $status_report = [];
      $has_warnings = FALSE;
      $has_errors = FALSE;

      foreach ($requirements as $requirement_key => $requirement) {
        $status_report[$requirement_key] = [
          'title' => (string) $requirement['title'],
          'value' => isset($requirement['value']) ? (string) $requirement['value'] : '',
          'description' => isset($requirement['description']) ? (string) $requirement['description'] : '',
          'severity' => $requirement['severity'] ?? REQUIREMENT_OK,
          'severity_title' => $this->getSeverityTitle($requirement['severity'] ?? REQUIREMENT_OK),
        ];

        // Check for warnings and errors
        $severity = $requirement['severity'] ?? REQUIREMENT_OK;
        if ($severity == REQUIREMENT_WARNING) {
          $has_warnings = TRUE;
        } elseif ($severity == REQUIREMENT_ERROR) {
          $has_errors = TRUE;
        }
      }

      $total_items = count($status_report);
      $message = $this->t('Successfully retrieved system status with @count items. Warnings: @warnings, Errors: @errors', [
        '@count' => $total_items,
        '@warnings' => $has_warnings ? 'Yes' : 'No',
        '@errors' => $has_errors ? 'Yes' : 'No',
      ]);

      return ExecutableResult::success($message, [
        'status_report' => $status_report,
        'requirements' => $requirements,
        'has_warnings' => $has_warnings,
        'has_errors' => $has_errors,
      ]);

    } catch (\Exception $e) {
      return ExecutableResult::failure($this->t('Error retrieving system status: @error', [
        '@error' => $e->getMessage()
      ]), [
        'status_report' => [],
        'requirements' => [],
        'has_warnings' => FALSE,
        'has_errors' => FALSE,
      ]);
    }
  }

  /**
   * Get human-readable severity title.
   *
   * @param int $severity
   *   The severity level.
   *
   * @return string
   *   The severity title.
   */
  private function getSeverityTitle(int $severity): string {
    switch ($severity) {
      case REQUIREMENT_INFO:
        return 'Info';
      case REQUIREMENT_OK:
        return 'OK';
      case REQUIREMENT_WARNING:
        return 'Warning';
      case REQUIREMENT_ERROR:
        return 'Error';
      default:
        return 'Unknown';
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    $account = $account ?? \Drupal::currentUser();
    // Check if user has permission to access administration pages
    $access = AccessResult::allowedIfHasPermission($account, 'administer site configuration');
    return $return_as_object ? $access : $access->isAllowed();
  }

}
