<?php

declare(strict_types=1);

namespace Drupal\tool_user\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\TypedData\InputDefinition;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ToolBase;
use Drupal\tool\TypedInputs;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the user add role tool.
 */
#[Tool(
  id: 'user_add_role',
  label: new TranslatableMarkup('User Add Role'),
  description: new TranslatableMarkup('Add a role to a user account.'),
  input_definitions: [
    'user' => new InputDefinition(
      data_type: 'entity:user',
      label: new TranslatableMarkup("User"),
      required: TRUE,
      description: new TranslatableMarkup("The user entity to add the role to.")
    ),
    'role_id' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Role ID"),
      required: TRUE,
      description: new TranslatableMarkup("The machine name of the role to add (e.g., editor, administrator).")
    ),
  ],
  output_definitions: [
    'user' => new ContextDefinition(
      data_type: 'entity:user',
      label: new TranslatableMarkup("User"),
      description: new TranslatableMarkup("The updated user entity.")
    ),
  ],
)]
class UserAddRole extends ToolBase {

  /**
   * Constructs a UserAddRole object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\tool\TypedInputs $typed_inputs
   *   The typed inputs service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    array $configuration,
          $plugin_id,
          $plugin_definition,
    TypedInputs $typed_inputs,
    protected EntityTypeManagerInterface $entityTypeManager
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $typed_inputs);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('typed.inputs'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    [
      'user' => $user,
      'role_id' => $role_id,
    ] = $values;

    try {
      if (!$user instanceof UserInterface) {
        return ExecutableResult::failure($this->t('Invalid user entity.'));
      }

      // Check if role exists
      $role_storage = $this->entityTypeManager->getStorage('user_role');
      $role = $role_storage->load($role_id);

      if (!$role) {
        return ExecutableResult::failure($this->t('Role "@role" does not exist.', [
          '@role' => $role_id
        ]));
      }

      // Check if user already has the role
      if ($user->hasRole($role_id)) {
        return ExecutableResult::success($this->t('User "@username" already has the "@role" role.', [
          '@username' => $user->getAccountName(),
          '@role' => $role->label()
        ]), ['user' => $user]);
      }

      // Add the role
      $user->addRole($role_id);
      $user->save();

      return ExecutableResult::success($this->t('Successfully added "@role" role to user "@username".', [
        '@role' => $role->label(),
        '@username' => $user->getAccountName()
      ]), ['user' => $user]);

    } catch (\Exception $e) {
      return ExecutableResult::failure($this->t('Error adding role to user: @message', [
        '@message' => $e->getMessage()
      ]));
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    $account = $account ?? \Drupal::currentUser();

    // Check if user has permission to assign roles
    // todo: confirm this is the right role to use vs 'administer roles and permissions'
    $access_result = AccessResult::allowedIfHasPermission($account, 'administer users');

    return $return_as_object ? $access_result : $access_result->isAllowed();
  }

}
