<?php

declare(strict_types=1);

namespace Drupal\tool_content\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ToolBase;
use Drupal\tool\TypedInputs;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the entity create tool.
 */
#[Tool(
  id: 'entity_create',
  label: new TranslatableMarkup('Entity Create'),
  description: new TranslatableMarkup('Create a new entity with field values.'),
  input_definitions: [],
  output_definitions: [
    'created_entity' => new ContextDefinition(
      data_type: 'entity',
      label: new TranslatableMarkup("Created Entity"),
      description: new TranslatableMarkup("The unsaved entity that was created.")
    ),
  ],
  deriver: 'Drupal\tool_content\Plugin\tool\Tool\Deriver\EntityCreateDeriver',
)]
class EntityCreate extends ToolBase {

  /**
   * Constructs a new EntityCreate instance.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin ID.
   * @param array $plugin_definition
   *   The plugin definition.
   * @param TypedInputs $typed_inputs
   *   The typed inputs service.
   * @param EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager service.
   */
  public function __construct(
    array $configuration,
          $plugin_id,
          $plugin_definition,
          TypedInputs $typed_inputs,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EntityFieldManagerInterface $entityFieldManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $typed_inputs);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('typed.inputs'),
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    // Extract entity type from plugin ID
    $entity_type_id = $this->getDerivativeId();

    try {
      // Create a new entity
      $entity_storage = $this->entityTypeManager->getStorage($entity_type_id);

      $entity_values = [];

      // Add field values from inputs
      foreach ($values as $field_name => $field_value) {
        if (!empty($field_value)) {
          $entity_values[$field_name] = $field_value;
        }
      }

      $entity = $entity_storage->create($entity_values);

      return ExecutableResult::success($this->t('Successfully created unsaved @type entity', [
        '@type' => $entity_type_id
      ]), ['created_entity' => $entity]);

    } catch (\Exception $e) {
      return ExecutableResult::failure($this->t('Error creating @type entity: @message', [
        '@type' => $entity_type_id,
        '@message' => $e->getMessage()
      ]));
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    $entity_type_id = $this->getDerivativeId();
    $account = $account ?? \Drupal::currentUser();
    // Get entity type definition
    $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
    $bundle_key = $entity_type->getKey('bundle');
    /**
     * @var \Drupal\Core\Entity\EntityAccessControlHandlerInterface $access_handler
     */
    $access_handler = $this->entityTypeManager->getHandler($entity_type_id, 'access');
    $access_result = $access_handler->createAccess($values[$bundle_key], $account, [], TRUE);

    return $return_as_object ? $access_result : $access_result->isAllowed();
  }

}
