<?php

declare(strict_types=1);

namespace Drupal\tool_content\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\TypedData\InputDefinition;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ToolBase;

/**
 * Plugin implementation of the field set value tool.
 */
#[Tool(
  id: 'field_set_value',
  label: new TranslatableMarkup('Field Set Value'),
  description: new TranslatableMarkup('Set a field value on an entity.'),
  input_definitions: [
    'entity' => new InputDefinition(
      data_type: 'entity',
      label: new TranslatableMarkup("Entity"),
      description: new TranslatableMarkup("The entity to modify.")
    ),
    'field_name' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Field Name"),
      description: new TranslatableMarkup("The machine name of the field to set.")
    ),
    'value' => new InputDefinition(
      data_type: 'any',
      label: new TranslatableMarkup("Value"),
      required: FALSE,
      description: new TranslatableMarkup("The value(s) to set for the field. Can be a single value or multiple values for multi-value fields. Should follow accepted field definition schema."),
    ),
  ],
  output_definitions: [
    'updated_entity' => new ContextDefinition(
      data_type: 'entity',
      label: new TranslatableMarkup("Updated Entity"),
      description: new TranslatableMarkup("The entity with the updated field value.")
    ),
  ],
)]
final class FieldSetValue extends ToolBase {

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    ['entity' => $entity, 'field_name' => $field_name, 'value' => $value] = $values;

    try {
      // Check if the field exists on the entity
      if (!$entity->hasField($field_name)) {
        return ExecutableResult::failure($this->t('Field "@field" does not exist on @type entity.', [
          '@field' => $field_name,
          '@type' => $entity->getEntityTypeId()
        ]));
      }

      // Get the field definition to understand the field structure
      $field_definition = $entity->getFieldDefinition($field_name);

      $cardinality = $field_definition->getFieldStorageDefinition()->getCardinality();

      // Todo: Deal with values as single value, single complex value, multiple values, etc.
      // Todo: Determine what to do with base fields.

      // Validate cardinality
      if ($cardinality !== -1 && is_array($value) && isset($value[0]) && count($value) > $cardinality) {
        return ExecutableResult::failure($this->t('Field "@field" accepts maximum @max values, @count provided.', [
          '@field' => $field_name,
          '@max' => $cardinality,
          '@count' => count($value)
        ]));
      }

      // Set the field value
      $entity->set($field_name, $value);

      // Validate the field value
      $violations = $entity->get($field_name)->validate();
      if ($violations->count() > 0) {
        $violation_messages = [];
        foreach ($violations as $violation) {
          $violation_messages[] = $violation->getMessage();
        }
        return ExecutableResult::failure($this->t('Field validation failed: @violations', [
          '@violations' => implode(', ', $violation_messages)
        ]));
      }

      return ExecutableResult::success($this->t('Successfully set field "@field" on @type entity.', [
        '@field' => $field_name,
        '@type' => $entity->getEntityTypeId()
      ]), ['updated_entity' => $entity]);

    } catch (\Exception $e) {
      return ExecutableResult::failure($this->t('Error setting field "@field": @message', [
        '@field' => $field_name,
        '@message' => $e->getMessage()
      ]));
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    $account = $account ?? \Drupal::currentUser();
    ['entity' => $entity, 'field_name' => $field_name, 'value' => $value] = $values;

    if (!$entity->get($field_name)->access('edit', $account)) {
      return $return_as_object ? AccessResult::forbidden() : AccessResult::forbidden()->isForbidden();
    }
    return $return_as_object ? AccessResult::allowed() : AccessResult::allowed()->isAllowed();
  }

}
