<?php

declare(strict_types=1);

namespace Drupal\tool_entity\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\TypedData\InputDefinition;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ConditionToolBase;
use Drupal\tool\Tool\ToolBase;
use Drupal\tool\TypedInputs;
use Symfony\Component\DependencyInjection\ContainerInterface;

// @todo: Tools that check conditions may utilize an interface to allow automatically generating conditions.
/**
 * Plugin implementation of the entity bundle exists tool.
 */
#[Tool(
  id: 'entity_bundle_exists',
  label: new TranslatableMarkup('Entity Bundle Exists'),
  description: new TranslatableMarkup('Check if an entity type and bundle combination exists.'),
  input_definitions: [
    'entity_type' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Entity Type"),
      description: new TranslatableMarkup("The entity type machine name (e.g., node, user, taxonomy_term).")
    ),
    'bundle' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Bundle"),
      description: new TranslatableMarkup("The bundle machine name (e.g., article, page).")
    ),
  ],
  // output_definitions automatically inherited from ConditionToolBase
)]
class EntityBundleExists extends ConditionToolBase {

  /**
   * Constructs a new EntityBundleExists instance.
   *
   * @param array $configuration
   *  The plugin configuration.
   * @param $plugin_id
   *  The plugin ID.
   * @param $plugin_definition
   *  The plugin definition.
   * @param \Drupal\tool\TypedInputs $typed_inputs
   *  The typed inputs service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *  The entity type manager service.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $bundleInfo
   *  The entity type bundle info service.
   */
  public function __construct(
    array $configuration,
          $plugin_id,
          $plugin_definition,
    TypedInputs $typed_inputs,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EntityTypeBundleInfoInterface $bundleInfo
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $typed_inputs);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static($configuration, $plugin_id, $plugin_definition,
      $container->get('typed.inputs'),
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    ['entity_type' => $entity_type, 'bundle' => $bundle] = $values;
    try {
      $entity_type_definition = $this->entityTypeManager->getDefinition($entity_type, FALSE);
      if (!$entity_type_definition) {
        return ExecutableResult::success($this->t('Entity type @type does not exist', [
          '@type' => $entity_type
        ]), ['result' => FALSE]);
      }

      $bundles = $this->bundleInfo->getBundleInfo($entity_type);
      $exists = isset($bundles[$bundle]);

      return ExecutableResult::success($this->t('Entity type @type with bundle @bundle @status', [
        '@type' => $entity_type,
        '@bundle' => $bundle,
        '@status' => $exists ? 'exists' : 'does not exist'
      ]), ['result' => $exists]);
    } catch (\Exception $e) {
      return ExecutableResult::failure($this->t('Error checking entity/bundle: @message', [
        '@message' => $e->getMessage()
      ]));
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    // todo Come back to.
    return $return_as_object ? AccessResult::allowed() : AccessResult::allowed()->isAllowed();
  }

}
