<?php

declare(strict_types=1);

namespace Drupal\tool_entity\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\TypedDataManagerInterface;
use Drupal\field\FieldConfigInterface;
use Drupal\tool\Tool\ExplainerToolInterface;
use Drupal\tool\TypedData\InputDefinition;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ToolBase;
use Drupal\tool\TypedInputs;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the entity field definitions tool.
 */
#[Tool(
  id: 'entity_bundle_field_definitions',
  label: new TranslatableMarkup('Entity Bundle Field Definitions'),
  description: new TranslatableMarkup('Get field definitions and value schema for a given entity type and bundle.'),
  input_definitions: [
    'entity_type_id' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Entity Type ID"),
      description: new TranslatableMarkup("The entity type ID (e.g., node, user, taxonomy_term).")
    ),
    'bundle' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Bundle"),
      description: new TranslatableMarkup("The bundle name (e.g., article, page for nodes).")
    ),
  ],
  output_definitions: [
    'field_definitions' => new ContextDefinition(
      data_type: 'map',
      label: new TranslatableMarkup("Field Definitions"),
      description: new TranslatableMarkup("Array of field definitions with metadata and example values.")
    ),
  ],
)]
class EntityBundleFieldDefinitions extends ToolBase implements ExplainerToolInterface {

  /**
   * Constructs a new EntityBundleFieldDefinitions instance.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin ID.
   * @param array $plugin_definition
   *   The plugin definition.
   * @param TypedInputs $typed_inputs
   *   The typed inputs service.
   * @param EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager service.
   * @param EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param EntityTypeBundleInfoInterface $entityTypeBundleInfo
   *   The entity type bundle info service.
   */
  public function __construct(
    array $configuration,
          $plugin_id,
          $plugin_definition,
          TypedInputs $typed_inputs,
    protected EntityFieldManagerInterface $entityFieldManager,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EntityTypeBundleInfoInterface $entityTypeBundleInfo,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $typed_inputs);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('typed.inputs'),
      $container->get('entity_field.manager'),
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    ['entity_type_id' => $entity_type_id, 'bundle' => $bundle] = $values;

    try {
      // Validate entity type exists
      if (!$this->entityTypeManager->hasDefinition($entity_type_id)) {
        return ExecutableResult::failure($this->t('Entity type "@type" does not exist.', [
          '@type' => $entity_type_id
        ]));
      }

      // Validate bundle exists
      $bundle_info = $this->entityTypeBundleInfo->getBundleInfo($entity_type_id);
      if (!isset($bundle_info[$bundle])) {
        return ExecutableResult::failure($this->t('Bundle "@bundle" does not exist for entity type "@type".', [
          '@bundle' => $bundle,
          '@type' => $entity_type_id
        ]));
      }

      // Get field definitions
      $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle);

      $field_data = [];
      $label_key = $this->entityTypeManager->getDefinition($entity_type_id)->getKey('label');

      foreach ($field_definitions as $field_name => $field_definition) {

        if (!$field_definition instanceof FieldConfigInterface && $field_name !== $label_key) {
          continue; // Skip if not a valid field definition
        }
        $field_data[$field_name] = $this->getFieldInfo($field_definition);
      }

      return ExecutableResult::success($this->t('Found @count field definitions for @type:@bundle', [
        '@count' => count($field_data),
        '@type' => $entity_type_id,
        '@bundle' => $bundle
      ]), ['field_definitions' => $field_data]);

    } catch (\Exception $e) {
      return ExecutableResult::failure($this->t('Error retrieving field definitions: @message', [
        '@message' => $e->getMessage()
      ]));
    }
  }

  /**
   * Get field information including metadata and example values.
   */
  private function getFieldInfo(FieldDefinitionInterface $field_definition): array {
    $field_type = $field_definition->getType();
    $settings = $field_definition->getSettings();

    $info = [
      'name' => $field_definition->getName(),
      'label' => $field_definition->getLabel(),
      'description' => $field_definition->getDescription(),
      'type' => $field_type,
      'required' => $field_definition->isRequired(),
      'cardinality' => $field_definition->getFieldStorageDefinition()->getCardinality(),
      'settings' => $settings,
      'value_schema' => $this->getFieldSchema($field_definition),
    ];

    return $info;
  }

  /**
   * Get the field schema from the field item definition.
   */
  private function getFieldSchema(FieldDefinitionInterface $field_definition): array {
    try {
      // Get the field item definition
      $item_definition = $field_definition->getItemDefinition();

      // Get the property definitions from the field item
      $property_definitions = $item_definition->getPropertyDefinitions();

      $schema = [];
      foreach ($property_definitions as $property_name => $property_definition) {
        if ($property_definition->isComputed()) {
          continue; // Skip computed properties
        }
        $schema[$property_name] = [
          'type' => $property_definition->getDataType(),
          'label' => (string) $property_definition->getLabel(),
          'description' => (string) $property_definition->getDescription(),
          'required' => $property_definition->isRequired(),
        ];

        // Add constraints if any
        $constraints = $property_definition->getConstraints();
        if (!empty($constraints)) {
          $schema[$property_name]['constraints'] = $constraints;
        }

        // Add settings if any
        $settings = $property_definition->getSettings();
        if (!empty($settings)) {
          $schema[$property_name]['settings'] = $settings;
        }
      }

      return $schema;
    } catch (\Exception $e) {
      return ['error' => 'Could not retrieve field schema: ' . $e->getMessage()];
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    // todo Come back to.
    return $return_as_object ? AccessResult::allowed() : AccessResult::allowed()->isAllowed();
  }

}
