<?php

declare(strict_types=1);

namespace Drupal\tool_entity\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\tool\TypedData\InputDefinition;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ConditionToolBase;
// @todo: Tools that check conditions may utilize an interface to allow automatically generating conditions.
/**
 * Plugin implementation of the field storage result tool.
 */
#[Tool(
  id: 'field_storage_result',
  label: new TranslatableMarkup('Field Storage Exists'),
  description: new TranslatableMarkup('Check if a field storage result for an entity type.'),
  input_definitions: [
    'entity_type' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Entity Type"),
      description: new TranslatableMarkup("The entity type machine name."),
      constraints: [
        "PluginExists" => [
          "manager" => "entity_type.manager",
          "interface" =>  ContentEntityInterface::class
        ],
      ],
    ),
    'field_name' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Field Name"),
      description: new TranslatableMarkup("The field machine name.")
    ),
  ],
  // output_definitions automatically inherited from ConditionToolBase
)]
class FieldStorageExists extends ConditionToolBase {

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    ['entity_type' => $entity_type, 'field_name' => $field_name] = $values;
    try {
      $field_storage = FieldStorageConfig::loadByName($entity_type, $field_name);
      $result = $field_storage !== NULL;

      return ExecutableResult::success($this->t('Field storage @field for @type @status', [
        '@field' => $field_name,
        '@type' => $entity_type,
        '@status' => $result ? 'result' : 'does not exist'
      ]), ['result' => $result]);
    } catch (\Exception $e) {
      return ExecutableResult::failure($this->t('Error checking field storage: @message', [
        '@message' => $e->getMessage()
      ]));
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    // todo Come back to.
    return $return_as_object ? AccessResult::allowed() : AccessResult::allowed()->isAllowed();
  }

}
