<?php

declare(strict_types=1);

namespace Drupal\tool_system\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\TypedData\InputDefinition;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ToolBase;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the log message tool.
 */
#[Tool(
  id: 'log_message',
  label: new TranslatableMarkup('Log Message'),
  description: new TranslatableMarkup('Log a message to the Drupal logging system.'),
  input_definitions: [
    'message' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Message"),
      description: new TranslatableMarkup("The message to log.")
    ),
    'level' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Log Level"),
      description: new TranslatableMarkup("The log level (emergency, alert, critical, error, warning, notice, info, debug)."),
      default_value: 'info',
      constraints: [
        'Choice' => [
          'choices' => ['emergency', 'alert', 'critical', 'error', 'warning', 'notice', 'info', 'debug'],
          'message' => new TranslatableMarkup('The log level must be one of: emergency, alert, critical, error, warning, notice, info, debug.'),
        ],
      ],
    ),
    'channel' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Channel"),
      description: new TranslatableMarkup("The log channel (defaults to 'tool')."),
      default_value: 'tool',
      required: false,
    ),
    'context' => new InputDefinition(
      data_type: 'map',
      label: new TranslatableMarkup("Context"),
      description: new TranslatableMarkup("Additional context data to include with the log message."),
      required: false,
    ),
  ],
  output_definitions: [
    'logged' => new ContextDefinition(
      data_type: 'boolean',
      label: new TranslatableMarkup("Logged"),
      description: new TranslatableMarkup("Whether the message was successfully logged.")
    ),
  ],
)]
final class LogMessage extends ToolBase {

  public function __construct(
    array $configuration,
          $plugin_id,
          $plugin_definition,
    protected \Drupal\tool\TypedInputs $typedInputs,
    protected LoggerInterface $logger
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $typedInputs);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('typed.inputs'),
      $container->get('logger.factory')->get('tool')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    [
      'message' => $message,
      'level' => $level,
      'channel' => $channel,
      'context' => $context
    ] = $values + ['context' => []];

    try {
      // Get logger for the specified channel if different from default
      $logger = $this->logger;
      if ($channel !== 'tool') {
        $logger = \Drupal::service('logger.factory')->get($channel);
      }

      // Ensure context is an array
      $context = $context ?? [];
      if (!is_array($context)) {
        $context = ['data' => $context];
      }

      // Log the message using the appropriate level
      switch ($level) {
        case 'emergency':
          $logger->emergency($message, $context);
          break;
        case 'alert':
          $logger->alert($message, $context);
          break;
        case 'critical':
          $logger->critical($message, $context);
          break;
        case 'error':
          $logger->error($message, $context);
          break;
        case 'warning':
          $logger->warning($message, $context);
          break;
        case 'notice':
          $logger->notice($message, $context);
          break;
        case 'info':
          $logger->info($message, $context);
          break;
        case 'debug':
          $logger->debug($message, $context);
          break;
        default:
          $logger->info($message, $context);
      }

      return ExecutableResult::success($this->t('Successfully logged @level message to @channel channel.', [
        '@level' => $level,
        '@channel' => $channel
      ]), ['logged' => true]);

    } catch (\Exception $e) {
      return ExecutableResult::failure($this->t('Error logging message: @message', [
        '@message' => $e->getMessage()
      ]), ['logged' => false]);
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    $account = $account ?? \Drupal::currentUser();
    $access = AccessResult::allowedIfHasPermission($account, 'use log_message tool');
    return $return_as_object ? $access : $access->isAllowed();
  }

}
