<?php

declare(strict_types=1);

namespace Drupal\tool_system\Plugin\tool\Tool;

use Drupal\Component\Render\PlainTextOutput;
use Drupal\Component\Utility\EmailValidatorInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Utility\Token;
use Drupal\tool\TypedData\InputDefinition;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ToolBase;
use Drupal\tool\TypedInputs;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the send email tool.
 */
#[Tool(
  id: 'send_email',
  label: new TranslatableMarkup('Send Email'),
  description: new TranslatableMarkup('Send an email with specified recipient, subject and body.'),
  input_definitions: [
    'recipient' => new InputDefinition(
      data_type: 'email',
      label: new TranslatableMarkup("Recipient"),
      description: new TranslatableMarkup("The email address of the recipient.")
    ),
    'subject' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Subject"),
      description: new TranslatableMarkup("The subject of the message.")
    ),
    'message' => new InputDefinition(
      data_type: 'text',
      label: new TranslatableMarkup("Message"),
      description: new TranslatableMarkup("The message that should be sent.")
    ),
    'token_data' => new InputDefinition(
      data_type: 'entity', // todo: Create a token data type?
      label: new TranslatableMarkup("Token Data"),
      required: FALSE,
      multiple: TRUE,
      description: new TranslatableMarkup("The associative array of data for token replacements, avoid using 'recipient', 'subject' and 'message' as keys as they are already used."),
    ),
  ],
)]
class SendEmail extends ToolBase {

  /**
   * Constructs an EmailAction object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Utility\Token $token
   *   The token service.
   * @param \Drupal\Core\Mail\MailManagerInterface $mailManager
   *   The mail manager.
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager.
   * @param \Drupal\Component\Utility\EmailValidatorInterface $emailValidator
   *   The email validator.
   */
  public function __construct(
    array $configuration,
          $plugin_id,
          $plugin_definition,
    TypedInputs $typed_inputs,
    protected Token $token,
    protected MailManagerInterface $mailManager,
    protected LanguageManagerInterface $languageManager,
    protected EmailValidatorInterface $emailValidator
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $typed_inputs);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static($configuration, $plugin_id, $plugin_definition,
      $container->get('typed.inputs'),
      $container->get('token'),
      $container->get('plugin.manager.mail'),
      $container->get('language_manager'),
      $container->get('email.validator')
    );
  }

  /**
   * Executes the 'send email' tool.
   *
   * @param $recipient
   *   The email address of the recipient.
   * @param $subject
   *   The subject of the email.
   * @param $message
   *   The message body of the email.
   * @param $token_data
   *   An associative array of data for token replacements.
   *
   * @return ExecutableResult
   */
  protected function doExecute(array $values): ExecutableResult {
    ['recipient' => $recipient, 'subject' => $subject, 'message' => $message, 'token_data' => $token_data] = $values;
    $data = [
      'subject' => $subject,
      'message' => $message,
    ] + (array) $token_data;

    // Todo Token replacement prior to context validation or during set
    $data['recipient'] = PlainTextOutput::renderFromHtml($this->token->replace($recipient, $data));
    // If the recipient is a registered user with a language preference, use
    // the recipient's preferred language. Otherwise, use the system default
    // language.
    $recipient_accounts = \Drupal::entityTypeManager()->getStorage('user')->loadByProperties(['mail' => $data['recipient']]);
    $recipient_account = reset($recipient_accounts);
    $langcode = $recipient_account ? $recipient_account->getPreferredLangcode() : $this->languageManager->getDefaultLanguage()->getId();
    $message = $this->mailManager->mail('system', 'action_send_email', $data['recipient'], $langcode, ['context' => $data]);
    // Error logging is handled by \Drupal\Core\Mail\MailManager::mail().
    if ($message['result']) {
      return ExecutableResult::success($this->t('Sent email to %recipient', ['%recipient' => $recipient]));
    }
    return ExecutableResult::failure($this->t('Failed to send email to %recipient', ['%recipient' => $recipient]));
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    $account = $account ?? \Drupal::currentUser();
    $access = AccessResult::allowedIfHasPermission($account, 'use send_email tool');
    return $return_as_object ? $access : $access->isAllowed();
  }

}
