<?php

declare(strict_types=1);

namespace Drupal\tool;

use Drupal\Component\Plugin\Exception\ContextException;
use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Form\SubformStateInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Plugin\ContextAwarePluginTrait;
use Drupal\Core\TypedData\TypedData;
use Drupal\Core\TypedData\TypedDataTrait;
use Drupal\tool\Exception\InputException;
use Drupal\tool\TypedData\InputDefinitionInterface;
use Drupal\tool\TypedData\SchemaWidget\TypedDataSchemaWidgetInterface;
use Drupal\tool\TypedData\SchemaWidget\TypedDataSchemaWidgetPluginManager;
use Drupal\Component\Plugin\ConfigurableInterface;
use Drupal\Component\Plugin\Context\ContextInterface as ComponentContextInterface;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheableDependencyInterface;
use Drupal\Core\Plugin\Context\Context;
use Drupal\Core\Plugin\Context\ContextInterface;
use Symfony\Component\Validator\ConstraintViolationList;
use Symfony\Component\Validator\ConstraintViolationListInterface;

/**
 * @todo Add class description.
 */
final class TypedInputs implements TypedInputsInterface {

  use DependencySerializationTrait;
  use TypedDataTrait;

  const STORAGE_CONFIGURATION = 'config';
  const STORAGE_INPUT = 'input';

  /**
   * The input definitions for this plugin.
   *
   * @var \Drupal\tool\TypedData\InputDefinitionInterface[]
   */
  protected array $inputDefinitions = [];

  /**
   * The storage for input and configuration values.
   *
   * @var array
   */
  protected array $storage = [];

  /**
   * The form widget instances for the input definitions.
   *
   * @var \Drupal\tool\TypedData\SchemaWidget\TypedDataSchemaWidgetInterface[]|null
   */
  protected ?array $formWidgetInstances = null;

  /**
   * Constructs an InputHandler object.
   */
  public function __construct(
    private readonly TypedDataSchemaWidgetPluginManager $schemaWidgetManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getConfiguration(): array {
    $configuration = [];
    foreach (array_keys($this->getInputDefinitions()) as $name) {
      $configuration[$name] = $this->getConfigurationValue($name);
    }
    return $configuration;
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $configuration): self {
    $default_values = $this->defaultConfiguration();
    foreach (array_keys($this->getInputDefinitions()) as $name) {
      // todo: decide what to do here and with default values.
      if (array_key_exists($name, $configuration)) {
        $this->setConfigurationValue($name, $configuration[$name], TRUE);
      }
      else {
        if (array_key_exists($name, $default_values)) {
          $this->setConfigurationValue($name, $default_values[$name], TRUE);
        }
      }
    }
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    $default_values = [];

    foreach ($this->getInputDefinitions() as $name => $definition) {
      $default_values[$name] = $definition->getDefaultValue();
    }
    return $default_values;
  }

  /**
   * {@inheritdoc}
   */
  public function getExecutableValues(): array {
    foreach ($this->getInputDefinitions(TRUE) as $name => $definition) {
      if ($definition->isLocked()) {
        $definition->setLocked(FALSE);
        // todo: test default value across all types(entity, field_definition, etc).
        $this->setInputValue($name, $definition->getDefaultValue());
        $definition->setLocked(TRUE);
      }
      elseif (isset($this->storage[self::STORAGE_INPUT][$name])) {
        // Input value already set, do nothing.
        continue;
      }
      elseif($this->hasConfigurationValue($name)) {
        $this->setInputValue($name, $this->getConfigurationValue($name)->getValue());
      }
      else {
        $this->setInputValue($name, $definition->getDefaultValue());
      }
    }
    $violations = $this->validateInputs();
    if ($this->validateInputs()->count() > 0) {
      throw new InputException('Input values are not valid.');
    }
    return $this->getInputValues() ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormWidgetInstance(string $name): TypedDataSchemaWidgetInterface {
    if (!isset($this->formWidgetInstances[$name])) {
      $data_definition = $this->getInputDefinition($name)->getDataDefinition();
      $schema_widget_definition = $this->schemaWidgetManager->getDefinitionFromDataType($data_definition);
      // todo add formAlter behavior.
      $this->formWidgetInstances[$name] = $this->schemaWidgetManager->createInstance($schema_widget_definition['id']);
    }
    return $this->formWidgetInstances[$name];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormElement(string $name, array &$element, SubformStateInterface $subform_state): array {
    $typed_data = $this->getTypedDataManager()->create($this->getInputDefinition($name)->getDataDefinition(), $value = NULL, $name);
    return $this->getFormWidgetInstance($name)->formElement($typed_data, $element, $subform_state);
  }

  /**
   * {@inheritdoc}
   */
  public function setInputDefinitions(array $input_definitions): self {
    // Validate each input definition.
    foreach ($input_definitions as $name => $definition) {
      if (!$definition instanceof InputDefinitionInterface) {
        throw new \InvalidInputException(sprintf('Input definition for "%s" must implement InputDefinitionInterface.', $name));
      }
    }
    $this->inputDefinitions = $input_definitions;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getInputDefinitions($include_locked = FALSE): array {
    if (!isset($this->inputDefinitions)) {
      throw new \LogicException('Input definitions are not set. Set them using setInputDefinitions() before calling this method.');
    }
    if (!$include_locked) {
      return array_filter($this->inputDefinitions, function ($definition) {
        return !$definition->isLocked();
      });
    }
    return $this->inputDefinitions;
  }

  /**
   * {@inheritdoc}
   */
  public function getInputDefinition(string $name): InputDefinitionInterface|InputException {
    $inputs = $this->getInputDefinitions(TRUE);
    if (isset($inputs[$name])) {
      return $inputs[$name];
    }
    throw new InputException(sprintf("The %s context is not a valid context.", $name));
  }
  /**
   * The data objects representing the input of this plugin.
   *
   * @var \Drupal\Core\Plugin\Context\ContextInterface[]
   */
  protected $input = [];

  /**
   * {@inheritdoc}
   */
  public function getInputs(): array {
    // Make sure all input objects are initialized.
    foreach ($this->getInputDefinitions() as $name => $definition) {
      $this->getInput($name);
    }
    return $this->storage[self::STORAGE_INPUT];
  }

  /**
   * {@inheritdoc}
   */
  public function getInput(string $name): ComponentContextInterface {
    // Check for a valid input value.
    if (!isset($this->storage[self::STORAGE_INPUT][$name])) {
      $this->storage[self::STORAGE_INPUT][$name] = new Context($this->getInputDefinition($name));
    }
    return $this->storage[self::STORAGE_INPUT][$name];
  }

  /**
   * {@inheritdoc}
   */
  public function setInput(string $name, ComponentContextInterface $input) {
    // Check that the input passed is an instance of our extended interface.
    if (!$input instanceof ContextInterface) {
      throw new InputException("Passed $name input must be an instance of \\Drupal\\Core\\Plugin\\Input\\ContextInterface");
    }
    $this->storage[self::STORAGE_INPUT][$name] = $input;
  }

  /**
   * {@inheritdoc}
   */
  public function getInputValues(): array {
    $values = [];
    foreach ($this->getInputDefinitions() as $name => $definition) {
      $values[$name] = isset($this->storage[self::STORAGE_INPUT][$name]) ? $this->storage[self::STORAGE_INPUT][$name]->getContextValue() : NULL;
    }
    return $values;
  }

  /**
   * {@inheritdoc}
   */
  public function getInputValue(string $name): mixed {
    return $this->getInput($name)->getContextValue();
  }

  /**
   * {@inheritdoc}
   */
  public function setInputValue(string $name, $value): self {
    if ($this->getInputDefinition($name)->isLocked()) {
      throw new InputException(sprintf("The %s input is locked and cannot be changed.", $name));
    }
    $this->setInput($name, Context::createFromContext($this->getInput($name), $value));
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getInputMapping(): array {
    // Determine how to work with auto schema mapping.
   return [];
  }

  /**
   * {@inheritdoc}
   */
  public function setInputMapping(array $input_mapping): self {
    // Determine how to work with auto schema mapping.
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function validateInputs(): ConstraintViolationList {
    $violations = new ConstraintViolationList();
    // @todo Implement the Symfony Validator component to let the validator
    //   traverse and set property paths accordingly.
    //   See https://www.drupal.org/project/drupal/issues/3153847.
    foreach ($this->getInputs() as $name => $input) {
      // todo Test for entity:*
      if ($input->getContextData()->getDataDefinition()->getDataType() === 'entity') {
        if (!$input->getContextData()->getValue() instanceof ContentEntityInterface) {
          $violations->addAll($input->validate());
        }
      }
      else {
        $violations->addAll($input->validate());
      }
    }
    return $violations;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheInputs(): array {
    $cache_inputs = [];
    // Applied inputs can affect the cache inputs when this plugin is
    // involved in caching, collect and return them.
    foreach ($this->getInputs() as $input) {
      /** @var \Drupal\Core\Cache\CacheableDependencyInterface $input */
      if ($input instanceof CacheableDependencyInterface) {
        $cache_inputs = Cache::mergeContexts($cache_inputs, $input->getCacheContexts());
      }
    }
    return $cache_inputs;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags(): array {
    $tags = [];
    // Applied inputs can affect the cache tags when this plugin is
    // involved in caching, collect and return them.
    foreach ($this->getInputs() as $input) {
      /** @var \Drupal\Core\Cache\CacheableDependencyInterface $input */
      if ($input instanceof CacheableDependencyInterface) {
        $tags = Cache::mergeTags($tags, $input->getCacheTags());
      }
    }
    return $tags;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheMaxAge(): int {
    $max_age = Cache::PERMANENT;
    // Applied inputs can affect the cache max age when this plugin is
    // involved in caching, collect and return them.
    foreach ($this->getInputs() as $input) {
      /** @var \Drupal\Core\Cache\CacheableDependencyInterface $input */
      if ($input instanceof CacheableDependencyInterface) {
        $max_age = Cache::mergeMaxAges($max_age, $input->getCacheMaxAge());
      }
    }
    return $max_age;
  }


  /**
   * Validates a configuration value against its definition and constraints.
   *
   * @param string $name
   *   The name of the configuration value to validate.
   * @param mixed $value
   *   The value to validate.
   *
   * @return \Symfony\Component\Validator\ConstraintViolationListInterface
   *   A list of constraint violations, if any.
   */
  public function validateConfigurationValue(string $name, mixed $value): ConstraintViolationListInterface {
    // Validate the configuration value against its context definition.
    $input_definition = $this->getInputDefinition($name);
    $typed_data = $this->getTypedDataManager()->create($input_definition->getDataDefinition(), $value, $name);
    return $typed_data->validate();
  }

  /**
   * Gets a specific configuration value by name.
   *
   * @param string $name
   *   The name of the configuration value.
   * @param bool $return_typed_data
   *   Whether to return the value as typed data.
   *
   * @return mixed
   *   The configuration value, or NULL if not set.
   */
  public function getConfigurationValue(string $name, bool $return_typed_data = FALSE): mixed {
    // Get the configuration value for the given name.
    if (isset($this->storage[self::STORAGE_CONFIGURATION][$name])) {
      return $return_typed_data ? $this->storage[self::STORAGE_CONFIGURATION][$name]->getContextValue() : $this->storage[self::STORAGE_CONFIGURATION][$name];
    }
    throw new PluginException("Configuration value not set for key: $name");
  }

  /**
   * Sets a configuration value.
   *
   * @param string $name
   *   The name of the configuration value.
   * @param mixed $value
   *   The value to set for the configuration.
   * @param bool $skip_validation
   *   Whether to skip validation for this value.
   *
   * @return $this
   *   The current instance for method chaining.
   */
  public function setConfigurationValue(string $name, mixed $value, $skip_validation = FALSE): self {
    // If isset in getInputDefinitions(), validate constraints.
    if (!$skip_validation) {
      $violations = $this->validateConfigurationValue($name, $value);
      if ($violations->count() > 0) {
        // If there are violations, throw an exception or handle them as needed.
        foreach ($violations as $violation) {
          throw new PluginException((string) $violation->getMessage());
        }
      }
    }
    if (isset($this->storage[self::STORAGE_CONFIGURATION][$name])) {
      $this->storage[self::STORAGE_CONFIGURATION][$name]->setValue($value);
    }
    else {
      $data_definition = $this->getInputDefinition($name)->getDataDefinition();
      $this->storage[self::STORAGE_CONFIGURATION][$name] = $this->getTypedDataManager()->create($data_definition, $value, $name);
    }
    return $this;
  }

  /**
   * Checks if a configuration value exists.
   *
   * @param string $name
   *   The name of the configuration value to check.
   *
   * @return bool
   *   TRUE if the configuration value exists, FALSE otherwise.
   */
  public function hasConfigurationValue(string $name): bool {
    return isset($this->storage[self::STORAGE_CONFIGURATION][$name]) && $this->storage[self::STORAGE_CONFIGURATION][$name] instanceof TypedData;
  }

  /**
   * {@inheritdoc}
   */
  public function getPluginDefinition() {
    // TODO: Implement getPluginDefinition() method.
  }
}
