<?php

namespace Drupal\tool_ai_connector\Plugin\AiFunctionCall;

use Drupal\ai\Exception\AiFunctionCallingExecutionError;
use Drupal\Core\Plugin\Context\EntityContextDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\Attribute\FunctionCall;
use Drupal\ai\Base\FunctionCallBase;
use Drupal\tool_ai_connector\Plugin\AiFunctionCall\Derivative\ToolPluginDeriver;
use Drupal\ai\Service\FunctionCalling\ExecutableFunctionCallInterface;
use Drupal\ai\Service\FunctionCalling\FunctionCallInterface;
use Drupal\ai\Utility\ContextDefinitionNormalizer;
use Drupal\tool\Tool\ToolInterface;
use Drupal\tool\Tool\ToolManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the action plugin function.
 */
#[FunctionCall(
  id: 'tool',
  function_name: 'tool',
  name: new TranslatableMarkup('Tool Plugin Wrapper'),
  description: '',
  deriver: ToolPluginDeriver::class
)]
class ToolPluginBase extends FunctionCallBase implements ExecutableFunctionCallInterface {


  protected array $values = [];
  /**
   * The status of the action execution.
   *
   * @var string|null
   */
  protected $executionStatus;

  /**
   * The error message, if any.
   *
   * @var string|null
   */
  protected $errorMessage;

  protected ?ToolInterface $toolPluginInstance;

  /**
   * Constructs a FunctionCall plugin.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\ai\Utility\ContextDefinitionNormalizer $context_definition_normalizer
   *   The context definition normalizer service.
   * @param \Drupal\Core\Action\ActionManager $action_manager
   *   The action manager service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected ContextDefinitionNormalizer $context_definition_normalizer,
    protected ToolManager $toolManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $context_definition_normalizer);
  }

  /**
   * Load from dependency injection container.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): FunctionCallInterface|static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('ai.context_definition_normalizer'),
      $container->get('plugin.manager.tool')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function execute() {
    try {
      foreach ($this->getPluginInstance()->getInputDefinitions() as $name => $definition) {
        if (isset($this->values[$name])) {
          $this->getPluginInstance()->setInputValue($name, $this->values[$name]);
        }
      }
      if ($this->getPluginInstance()->access()) {
        $this->getPluginInstance()->execute();
      }
      else {
        $this->errorMessage = 'Tool plugin access denied.';
      }
    }
    catch (\Exception $e) {
      $this->errorMessage = 'Tool plugin execution failed: ' . $e->getMessage();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getReadableOutput(): string {
    // todo move to unified artifact solution.
    $tempstore = \Drupal::service('tempstore.private')->get('ai_tool_artifacts');

    if (isset($this->errorMessage)) {
      $output = $this->errorMessage;
      $properties = [];

      foreach (\Drupal::service('ai.context_definition_normalizer')->normalize($this->getPluginInstance()->getInputDefinitions()) as $property) {
        $properties[$property->getName()] = $property->renderPropertyArray();
      }

      $object = [
        'type' => 'object',
        'properties' => $properties,
      ];
      $output .= "\nThe tool input schema is: " . json_encode($object);
    }
    else if ($this->getPluginInstance()?->getResult()?->isSuccess()) {
      $output = (string) $this->t('Tool Plugin executed successfully: @result',
        [
          '@result' => (string) $this->getPluginInstance()->getResult()->getMessage(),
        ]);
      foreach ($this->getPluginInstance()->getOutputDefinitions() as $key => $definition) {
        if ($definition instanceof EntityContextDefinition || $definition->getDataType() === 'entity') {
          // If the output is an entity, we can provide a link to it.
          $entity = $this->getPluginInstance()->getOutputValue($key);
          $output .= "\n";

          // Get the entity's langcode for the artifact
          $langcode = 'und'; // Default to 'undefined' if no language
          if (method_exists($entity, 'language')) {
            $langcode = $entity->language()->getId();
          }

          if ($entity->isNew()) {
            if (!isset($entity->ai_hash)) {
              $hash = substr(md5(serialize($entity)), 0, 6);
              $entity->ai_hash = $hash;
            }
            $artifact_key = "{{entity:" . $entity->getEntityTypeId() . ":new:" . $entity->ai_hash . ":" . $langcode . "}}";
            $output .= "An artifact has been created for the returned " . $key . " output.  To use this new entity for subsequent tool input arguments, pass the artifact string '" . $artifact_key . "' for entity inputs.\n";
          }
          else {
            $artifact_key = "{{entity:" . $entity->getEntityTypeId() . ":" . $entity->id() . ":" . $langcode . "}}";
            $output .= "An artifact has been created for the returned " . $key . " output.  To use this entity for subsequent tool input arguments, pass the artifact string '" . $artifact_key . "' for entity inputs.\n";
            $output .= "The artifact entity(" . $entity->getEntityTypeId() . ") id(" . $entity->id() . ") langcode(" . $langcode . ") and revision id(" . $entity->getRevisionId() . ")\n";
          }

          $artifact_key = str_replace('{{', 'artifact__', $artifact_key);
          $artifact_key = str_replace('}}', '', $artifact_key);
          $tempstore->delete($artifact_key);
          $tempstore->set($artifact_key, $entity);
        }
        else {
          $value = $this->getPluginInstance()->getOutputValue($key);
          $output .= "\n";
          $output .= "Output for " . $key . ": " . (is_array($value) ? json_encode($value) : $value);
        }
      }
    }
    else {
      $output = (string) $this->t('Tool Plugin executed unsuccessfully: @result',
        [
          '@result' => (string) $this->getPluginInstance()->getResult()->getMessage(),
        ]);
    }
    return $output;
  }

  /**
   * {@inheritdoc}
   */
  public function getContextDefinitions() {
    return $this->getPluginInstance()->getInputDefinitions();
  }

  /**
   * {@inheritdoc}
   */
  public function getContextDefinition($name) {
    return $this->getPluginInstance()->getInputDefinition($name);
  }

  /**
   * {@inheritdoc}
   */
  public function getContextValues() {
    return $this->values;
  }

  // todo potentially override every method from ContextAwarePluginInterface.

  /**
   * {@inheritdoc}
   */
  public function setContextValue($name, $value) {
    // If multiple, convert the value based on list, then convert each item.
    if ($this->getContextDefinition($name)->isMultiple()) {
      $value = $this->dataTypeConverterManager()->convert('list', $value);
      foreach ($value as $delta => $item) {
        $value[$delta] = $this->dataTypeConverterManager()->convert($this->getContextDefinition($name)->getDataType(), $item);
      }
    }
    else {
      $value = $this->dataTypeConverterManager()->convert($this->getContextDefinition($name)->getDataType(), $value);
    }
    $this->values[$name] = $value;
//    $this->getPluginInstance()->setInputValue($name, $value);
  }

  public function getPluginInstance() {
    if (!isset($this->toolPluginInstance)) {
      $this->toolPluginInstance = $this->toolManager->createInstance($this->getDerivativeId());
    }
    return $this->toolPluginInstance;
  }

  /**
   * {@inheritdoc}
   */
  public function validateContexts() {
    // todo: fix.
    return [];
    return $this->toolPluginInstance->validateInputs();
  }

}
