<?php

namespace Drupal\tool\Tool;

use Drupal\Component\Plugin\Definition\DerivablePluginDefinitionInterface;
use Drupal\Component\Plugin\Definition\PluginDefinition;
use Drupal\tool\TypedData\InputDefinitionInterface;
use Drupal\Core\Plugin\Context\ContextDefinitionInterface;
use Drupal\Component\Plugin\Exception\ContextException;
use Drupal\Core\StringTranslation\TranslatableMarkup;

class ToolDefinition extends PluginDefinition implements DerivablePluginDefinitionInterface {

  /**
   * The human-readable name.
   *
   * @var string
   */
  protected $label;

  /**
   * An optional description for advanced layouts.
   *
   * @var string
   */
  protected $description;
  /**
   * The input definitions for this plugin definition.
   *
   * @var \Drupal\tool\TypedData\InputDefinitionInterface[]
   */
  protected $inputDefinitions = [];

  /**
   * The output definitions for this plugin definition.
   *
   * @var \Drupal\Core\Plugin\Context\ContextDefinitionInterface
   */
  protected $outputDefinitions = [];

  /**
   * An array of form class names or FALSE, keyed by a string.
   *
   * @var array<string, string|false>
   */
  protected $forms = [];

  /**
   * Whether the tool is destructive and requires confirmation.
   *
   * @var bool
   */
  protected $destructive = false;

  /**
   * Input definition refiners configuration for dynamic input refinement.
   *
   * @var array
   */
  protected $inputDefinitionRefiners = [];

  /**
   * The name of the deriver of this layout definition, if any.
   *
   * @var string|null
   */
  protected $deriver;

  /**
   * @param array $definition
   *   An array of values from the attribute.
   */
  public function __construct(array $definition) {
    if (isset($definition['input_definitions'])) {
      foreach ($definition['input_definitions'] as $name => $input_definition) {
        $this->addInputDefinition($name, $input_definition);
      }
      unset($definition['input_definitions']);
    }

    if (isset($definition['output_definitions'])) {
      foreach ($definition['output_definitions'] as $name => $output_definition) {
        $this->addOutputDefinition($name, $output_definition);
      }
      unset($definition['output_definitions']);
    }

    if (isset($definition['input_definition_refiners'])) {
      $this->setInputDefinitionRefiners($definition['input_definition_refiners']);
      unset($definition['input_definition_refiners']);
    }

    foreach ($definition as $property => $value) {
      $this->set($property, $value);
    }
  }

  /**
   * Gets any arbitrary property.
   *
   * @param string $property
   *   The property to retrieve.
   *
   * @return mixed
   *   The value for that property, or NULL if the property does not exist.
   */
  public function get($property) {
    $value = $this?->{$property} ?? NULL;
    return $value;
  }

  /**
   * Sets a value to an arbitrary property.
   *
   * @param string $property
   *   The property to use for the value.
   * @param mixed $value
   *   The value to set.
   *
   * @return $this
   */
  public function set($property, $value) {
    if (property_exists($this, $property)) {
      $this->{$property} = $value;
    }
    return $this;
  }

  /**
   * Checks if an input definition exists.
   */
  public function hasInputDefinition($name) {
    return array_key_exists($name, $this->inputDefinitions);
  }

  /**
   * Gets all input definitions.
   */
  public function getInputDefinitions($include_locked = FALSE) {
    if ($include_locked) {
      return $this->inputDefinitions;
    }
    return array_filter($this->inputDefinitions, function($definition) {
      return !$definition->isLocked();
    });
  }

  /**
   * Gets a specific input definition.
   */
  public function getInputDefinition($name) {
    if ($this->hasInputDefinition($name)) {
      return $this->inputDefinitions[$name];
    }
    throw new ContextException($this->id() . " does not define a '$name' input");
  }

  /**
   * Adds an input definition.
   */
  public function addInputDefinition($name, InputDefinitionInterface $definition) {
    $this->inputDefinitions[$name] = $definition;
    return $this;
  }

  /**
   * Removes an input definition.
   */
  public function removeInputDefinition($name) {
    unset($this->inputDefinitions[$name]);
    return $this;
  }

  /**
   * Checks if an output definition exists.
   */
  public function hasOutputDefinition($name) {
    return array_key_exists($name, $this->outputDefinitions);
  }

  /**
   * Gets all output definitions.
   */
  public function getOutputDefinitions() {
    return $this->outputDefinitions;
  }

  /**
   * Gets a specific output definition.
   */
  public function getOutputDefinition($name) {
    if ($this->hasOutputDefinition($name)) {
      return $this->outputDefinitions[$name];
    }
    throw new ContextException($this->id() . " does not define a '$name' output");
  }

  /**
   * Adds an output definition.
   */
  public function addOutputDefinition($name, ContextDefinitionInterface $definition) {
    $this->outputDefinitions[$name] = $definition;
    return $this;
  }

  /**
   * Removes an output definition.
   */
  public function removeOutputDefinition($name) {
    unset($this->outputDefinitions[$name]);
    return $this;
  }

  /**
   * Gets the human-readable name of the layout definition.
   *
   * @return string|\Drupal\Core\StringTranslation\TranslatableMarkup
   *   The human-readable name of the layout definition.
   */
  public function getLabel() {
    return $this->label ?? '';
  }

  /**
   * Sets the human-readable name of the layout definition.
   *
   * @param string|\Drupal\Core\StringTranslation\TranslatableMarkup $label
   *   The human-readable name of the layout definition.
   *
   * @return $this
   */
  public function setLabel($label) {
    $this->label = $label;
    return $this;
  }

  /**
   * Gets the description of the layout definition.
   *
   * @return string|\Drupal\Core\StringTranslation\TranslatableMarkup
   *   The description of the layout definition.
   */
  public function getDescription() {
    return $this->description ?? '';
  }

  /**
   * Sets the description of the layout definition.
   *
   * @param string|\Drupal\Core\StringTranslation\TranslatableMarkup $description
   *   The description of the layout definition.
   *
   * @return $this
   */
  public function setDescription($description) {
    $this->description = $description;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getDeriver() {
    if (empty($this->deriver)) {
      return NULL;
    }
    return $this->deriver;
  }

  /**
   * {@inheritdoc}
   */
  public function setDeriver($deriver) {
    $this->deriver = $deriver;
    return $this;
  }

  /**
   * Gets the forms for this tool definition.
   *
   * @return array<string, string|false>
   *   An array of form class names or FALSE, keyed by a string.
   */
  public function getForms() {
    return $this->forms;
  }

  /**
   * Gets a specific form class name by its
   */
  public function getForm($name) {
    return $this->forms[$name] ?? NULL;
  }

  /**
   * Sets the forms for this tool definition.
   *
   * @param array<string, string|false> $forms
   *   An array of form class names or FALSE, keyed by a string.
   *
   * @return $this
   */
  public function setForms($forms) {
    $this->forms = $forms;
    return $this;
  }

  /**
   * Adds a form to this tool definition.
   *
   * @param string $name
   *   The form name.
   * @param string|false $form_class
   *   The form class name or FALSE.
   *
   * @return $this
   */
  public function addForm($name, $form_class) {
    $this->forms[$name] = $form_class;
    return $this;
  }

  /**
   * Removes a form from this tool definition.
   *
   * @param string $name
   *   The form name to remove.
   *
   * @return $this
   */
  public function removeForm($name) {
    unset($this->forms[$name]);
    return $this;
  }

  /**
   * Gets whether this tool is destructive.
   *
   * @return bool
   *   TRUE if the tool is destructive and requires confirmation.
   */
  public function isDestructive() {
    return $this->destructive;
  }

  /**
   * Sets whether this tool is destructive.
   *
   * @param bool $destructive
   *   TRUE if the tool is destructive and requires confirmation.
   *
   * @return $this
   */
  public function setDestructive($destructive) {
    $this->destructive = $destructive;
    return $this;
  }

  /**
   * Gets the input definition refiners configuration.
   *
   * @return array
   *   The input definition refiners configuration array.
   */
  public function getInputDefinitionRefiners() {
    return $this->inputDefinitionRefiners;
  }

  /**
   * Sets the input definition refiners configuration.
   *
   * @param array $input_definition_refiners
   *   The input definition refiners configuration array.
   *
   * @return $this
   *
   * @throws \Drupal\Component\Plugin\Exception\ContextException
   *   When input definition refiners reference non-existent input definitions.
   */
  public function setInputDefinitionRefiners($input_definition_refiners) {
    // Validate that all referenced inputs exist in input definitions.
    foreach ($input_definition_refiners as $input_name => $dependencies) {
      if (!$this->hasInputDefinition($input_name)) {
        throw new ContextException("Input definition refiner references non-existent input definition '{$input_name}'");
      }
      foreach ($dependencies as $dependency) {
        if (!$this->hasInputDefinition($dependency)) {
          throw new ContextException("Input definition refiner for '{$input_name}' references non-existent dependency '{$dependency}'");
        }
      }
    }

    $this->inputDefinitionRefiners = $input_definition_refiners;
    return $this;
  }

  /**
   * Gets input definition refiners for a specific input.
   *
   * @param string $input_name
   *   The input name to get input definition refiners for.
   *
   * @return array|null
   *   The array of input dependencies for the given input, or NULL if none.
   */
  public function getInputDefinitionRefiner($input_name) {
    return $this->inputDefinitionRefiners[$input_name] ?? NULL;
  }

  /**
   * Adds an input definition refiner for a specific input.
   *
   * @param string $input_name
   *   The input name to add an input definition refiner for.
   * @param array $dependencies
   *   The array of input dependencies.
   *
   * @return $this
   *
   * @throws \Drupal\Component\Plugin\Exception\ContextException
   *   When input definition refiners reference non-existent input definitions.
   */
  public function addInputDefinitionRefiner($input_name, array $dependencies) {
    // Validate that the input and its dependencies exist.
    if (!$this->hasInputDefinition($input_name)) {
      throw new ContextException("Input definition refiner references non-existent input definition '{$input_name}'");
    }
    foreach ($dependencies as $dependency) {
      if (!$this->hasInputDefinition($dependency)) {
        throw new ContextException("Input definition refiner for '{$input_name}' references non-existent dependency '{$dependency}'");
      }
    }

    $this->inputDefinitionRefiners[$input_name] = $dependencies;
    return $this;
  }

  /**
   * Removes an input definition refiner for a specific input.
   *
   * @param string $input_name
   *   The input name to remove the input definition refiner for.
   *
   * @return $this
   */
  public function removeInputDefinitionRefiner($input_name) {
    unset($this->inputDefinitionRefiners[$input_name]);
    return $this;
  }

}
