<?php

declare(strict_types=1);

namespace Drupal\tool_entity\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Tool\ExplainerToolInterface;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ToolBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the entity type list tool.
 */
#[Tool(
  id: 'entity_type_list',
  label: new TranslatableMarkup('Entity Type List'),
  description: new TranslatableMarkup('Get a list of all content entity types with their bundles'),
  input_definitions: [],
  output_definitions: [
    'entity_types' => new ContextDefinition(
      data_type: 'map',
      label: new TranslatableMarkup("Entity Types"),
      description: new TranslatableMarkup("Array of content entity types with their bundles")
    ),
  ],
)]
class EntityTypeList extends ToolBase implements ExplainerToolInterface, ContainerFactoryPluginInterface {

  /**
   * Constructs a new EntityTypeList instance.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin ID.
   * @param array $plugin_definition
   *   The typed inputs service.
   * @param EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager service.
   * @param EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param EntityTypeBundleInfoInterface $entityTypeBundleInfo
   *   The entity type bundle info service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected EntityFieldManagerInterface $entityFieldManager,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EntityTypeBundleInfoInterface $entityTypeBundleInfo,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_field.manager'),
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    try {
      $entity_types_data = [];
      $entity_type_definitions = $this->entityTypeManager->getDefinitions();

      foreach ($entity_type_definitions as $entity_type_id => $entity_type_definition) {
        // Only include content entities
        if (!$entity_type_definition->entityClassImplements('\Drupal\Core\Entity\ContentEntityInterface')) {
          continue;
        }

        $entity_type_info = [
          'id' => $entity_type_id,
          'label' => (string) $entity_type_definition->getLabel(),
          // 'class' => $entity_type_definition->getClass(),
          // 'base_table' => $entity_type_definition->getBaseTable(),
          // 'data_table' => $entity_type_definition->getDataTable(),
          // 'revision_table' => $entity_type_definition->getRevisionTable(),
          // 'revision_data_table' => $entity_type_definition->getRevisionDataTable(),
          'bundle_entity_type' => $entity_type_definition->getBundleEntityType(),
          'bundle_label' => (string) $entity_type_definition->getBundleLabel(),
          // 'admin_permission' => $entity_type_definition->getAdminPermission(),
          'keys' => $entity_type_definition->getKeys(),
          'bundles' => [],
        ];

        // Get bundle information
        $bundle_info = $this->entityTypeBundleInfo->getBundleInfo($entity_type_id);
        // todo Get translation info.
        foreach ($bundle_info as $bundle_id => $bundle_data) {
          $bundle_info_array = [
            'id' => $bundle_id,
            'label' => $bundle_data['label'],
          ];
          $entity_type_info['bundles'][$bundle_id] = $bundle_info_array;
        }

        $entity_types_data[$entity_type_id] = $entity_type_info;
      }

      return ExecutableResult::success($this->t('Found @count content entity types with property schemas', [
        '@count' => count($entity_types_data)
      ]), ['entity_types' => $entity_types_data]);
    }
    catch (\Exception $e) {
      return ExecutableResult::failure($this->t('Error retrieving entity types: @message', [
        '@message' => $e->getMessage()
      ]));
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    // Check if user has permission to view entity information
    $account ??= \Drupal::currentUser();
    // todo : decide on access control
    return $return_as_object ? AccessResult::allowed() : TRUE;
  }

}
