<?php

namespace Drupal\tool;

use Drupal\Component\Plugin\Context\ContextDefinitionInterface;
use Drupal\Component\Plugin\Context\ContextInterface;
use Drupal\Component\Plugin\Exception\ContextException;
use Drupal\Core\Plugin\Context\Context;
use Symfony\Component\Validator\ConstraintViolationList;

trait TypedOutputsTrait {

  /**
   * The data objects representing the context of this plugin.
   *
   * @var \Drupal\Component\Plugin\Context\ContextInterface[]
   */
  protected $outputs = [];

  /**
   * {@inheritdoc}
   */
  public function setOutputValue(string $name, mixed $value): self {
    $context = Context::createFromContext($this->getOutput($name), $value);
    $this->outputs[$name] = $context;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getOutputs(): array {
    // Make sure all context objects are initialized.
    foreach ($this->getOutputDefinitions() as $name => $definition) {
      $this->getOutput($name);
    }
    return $this->context;
  }

  /**
   * {@inheritdoc}
   */
  public function getOutput(string $name): ContextInterface {
    // Check for a valid context value.
    if (!isset($this->outputs[$name])) {
      $this->outputs[$name] = new Context($this->getOutputDefinition($name));
    }
    return $this->outputs[$name];
  }

  public function getOutputValue(string $name): mixed {
    if (!isset($this->outputs[$name])) {
      throw new ContextException(sprintf("The provided context '%s' is not valid.", $name));
    }
    return $this->outputs[$name]->getContextValue();
  }

  /**
   * {@inheritdoc}
   */
  public function getOutputValues(): array {
    $values = [];
    foreach ($this->getOutputDefinitions() as $name => $definition) {
      $values[$name] = isset($this->outputs[$name]) ? $this->outputs[$name]->getContextValue() : NULL;
    }
    return $values;
  }

  /**
   * {@inheritdoc}
   */
  public function getOutputDefinition(string $name): ContextDefinitionInterface {
    return $this->getPluginDefinition()->getOutputDefinition($name);
  }

  /**
   * {@inheritdoc}
   */
  public function getOutputDefinitions(): array {
    return $this->getPluginDefinition()->getOutputDefinitions();
  }

  /**
   * {@inheritdoc}
   */
  public function validateOutputs(): ConstraintViolationList {
    $violations = new ConstraintViolationList();
    // @todo Implement the Symfony Validator component to let the validator
    //   traverse and set property paths accordingly.
    //   See https://www.drupal.org/project/drupal/issues/3153847.
    foreach ($this->getOutputs() as $context) {
      $violations->addAll($context->validate());
    }
    return $violations;
  }

}
