<?php

declare(strict_types=1);

namespace Drupal\tool_entity\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\TypedData\InputDefinition;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ConditionToolBase;

use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the entity bundle exists tool.
 */
#[Tool(
  id: 'entity_bundle_exists',
  label: new TranslatableMarkup('Check if bundle exists'),
  description: new TranslatableMarkup('Check if an entity type and bundle combination exists.'),
  input_definitions: [
    'entity_type' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Entity Type"),
      description: new TranslatableMarkup("The entity type machine name (e.g., node, user, taxonomy_term).")
    ),
    'bundle' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Bundle"),
      description: new TranslatableMarkup("The bundle machine name (e.g., article, page).")
    ),
  ],
  // output_definitions automatically inherited from ConditionToolBase.
)]
class EntityBundleExists extends ConditionToolBase {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity type bundle info service.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  protected EntityTypeBundleInfoInterface $bundleInfo;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->bundleInfo = $container->get('entity_type.bundle.info');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    ['entity_type' => $entity_type, 'bundle' => $bundle] = $values;
    try {
      $entity_type_definition = $this->entityTypeManager->getDefinition($entity_type, FALSE);
      if (!$entity_type_definition) {
        return ExecutableResult::success($this->t('Entity type @type does not exist', [
          '@type' => $entity_type,
        ]), ['result' => FALSE]);
      }

      $bundles = $this->bundleInfo->getBundleInfo($entity_type);
      $exists = isset($bundles[$bundle]);

      return ExecutableResult::success($this->t('Entity type @type with bundle @bundle @status', [
        '@type' => $entity_type,
        '@bundle' => $bundle,
        '@status' => $exists ? 'exists' : 'does not exist',
      ]), ['result' => $exists]);
    }
    catch (\Exception $e) {
      return ExecutableResult::failure($this->t('Error checking entity/bundle: @message', [
        '@message' => $e->getMessage(),
      ]));
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, AccountInterface $account, bool $return_as_object = FALSE): bool|AccessResultInterface {
    // @todo Come back to.
    return $return_as_object ? AccessResult::allowed() : AccessResult::allowed()->isAllowed();
  }

}
