<?php

declare(strict_types=1);

namespace Drupal\tool_entity\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\TypedData\InputDefinition;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ConditionToolBase;

use Symfony\Component\DependencyInjection\ContainerInterface;

// @todo Tools that check conditions may utilize an interface to allow automatically generating conditions.
/**
 * Plugin implementation of the field exists tool.
 */
#[Tool(
  id: 'field_exists',
  label: new TranslatableMarkup('Check if field exists'),
  description: new TranslatableMarkup('Check if a field exists on an entity bundle.'),
  input_definitions: [
    'entity_type' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Entity Type"),
      description: new TranslatableMarkup("The entity type machine name."),
      constraints: [
        "PluginExists" => [
          "manager" => "entity_type.manager",
          "interface" => ContentEntityInterface::class,
        ],
      ],
    ),
    'bundle' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Bundle"),
      description: new TranslatableMarkup("The bundle machine name.")
    ),
    'field_name' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Field Name"),
      description: new TranslatableMarkup("The field machine name.")
    ),
  ],
  // output_definitions automatically inherited from ConditionToolBase.
)]
class FieldExists extends ConditionToolBase {

  /**
   * The entity field manager service.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected EntityFieldManagerInterface $fieldManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->fieldManager = $container->get('entity_field.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    ['entity_type' => $entity_type, 'bundle' => $bundle, 'field_name' => $field_name] = $values;
    try {
      $field_definitions = $this->fieldManager->getFieldDefinitions($entity_type, $bundle);
      $result = isset($field_definitions[$field_name]);

      return ExecutableResult::success($this->t('Field @field on @type:@bundle @status', [
        '@field' => $field_name,
        '@type' => $entity_type,
        '@bundle' => $bundle,
        '@status' => $result ? 'result' : 'does not exist',
      ]), ['result' => $result]);
    }
    catch (\Exception $e) {
      return ExecutableResult::failure($this->t('Error checking field: @message', [
        '@message' => $e->getMessage(),
      ]));
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, AccountInterface $account, bool $return_as_object = FALSE): bool|AccessResultInterface {
    // @todo Come back to.
    return $return_as_object ? AccessResult::allowed() : AccessResult::allowed()->isAllowed();
  }

}
