<?php

declare(strict_types=1);

namespace Drupal\tool_entity\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\content_translation\ContentTranslationManagerInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\TypedDataManagerInterface;
use Drupal\field\FieldConfigInterface;
use Drupal\tool\Tool\ExplainerToolInterface;
use Drupal\tool\TypedData\InputDefinition;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ToolBase;
use Drupal\tool\TypedData\MapInputDefinition;

use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the entity field definitions tool.
 */
#[Tool(
  id: 'entity_bundle_field_definitions',
  label: new TranslatableMarkup('Entity Bundle Field Definitions'),
  description: new TranslatableMarkup('Get field definitions entity type and bundle.  Usually used before entity type and bundle tools.'),
  input_definitions: [
    'entity_type_id' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Entity Type ID"),
      description: new TranslatableMarkup("The entity type ID (e.g., node, user, taxonomy_term).")
    ),
    'bundle' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Bundle"),
      description: new TranslatableMarkup("The bundle name (e.g., article, page for nodes).")
    ),
  ],
  output_definitions: [
    'field_definitions' => new ContextDefinition(
      data_type: 'map',
      label: new TranslatableMarkup("Field Definitions"),
      description: new TranslatableMarkup("Array of field definitions")
    ),
  ],
)]
class EntityBundleFieldDefinitions extends ToolBase implements ExplainerToolInterface, ContainerFactoryPluginInterface {

  /**
   * Constructs a new EntityBundleFieldDefinitions instance.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin ID.
   * @param array $plugin_definition
   *   The plugin definition.
   * @param EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager service.
   * @param EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param EntityTypeBundleInfoInterface $entityTypeBundleInfo
   *   The entity type bundle info service.
   * @param ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   * @param ContentTranslationManagerInterface|null $contentTranslationManager
   *   The content translation manager service.
   */
  public function __construct(
    array $configuration,
          $plugin_id,
          $plugin_definition,
    protected EntityFieldManagerInterface $entityFieldManager,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EntityTypeBundleInfoInterface $entityTypeBundleInfo,
    protected ModuleHandlerInterface $moduleHandler,
    protected ?ContentTranslationManagerInterface $contentTranslationManager = NULL,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_field.manager'),
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('module_handler'),
      $container->has('content_translation.manager') ? $container->get('content_translation.manager') : NULL
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    ['entity_type_id' => $entity_type_id, 'bundle' => $bundle] = $values;

    try {
      // Validate entity type exists
      if (!$this->entityTypeManager->hasDefinition($entity_type_id)) {
        return ExecutableResult::failure($this->t('Entity type "@type" does not exist.', [
          '@type' => $entity_type_id
        ]));
      }

      // Validate bundle exists
      $bundle_info = $this->entityTypeBundleInfo->getBundleInfo($entity_type_id);
      if (!isset($bundle_info[$bundle])) {
        return ExecutableResult::failure($this->t('Bundle "@bundle" does not exist for entity type "@type".', [
          '@bundle' => $bundle,
          '@type' => $entity_type_id
        ]));
      }

      // Get field definitions
      $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle);

      $field_data = [];
      $label_key = $this->entityTypeManager->getDefinition($entity_type_id)->getKey('label');

      foreach ($field_definitions as $field_name => $field_definition) {

        if (!$field_definition instanceof FieldConfigInterface && $field_name !== $label_key) {
          continue; // Skip if not a valid field definition
        }
        $field_data[$field_name] = $this->getFieldInfo($field_definition, $entity_type_id, $bundle);
      }

      return ExecutableResult::success($this->t('Found @count field definitions for @type:@bundle', [
        '@count' => count($field_data),
        '@type' => $entity_type_id,
        '@bundle' => $bundle
      ]), ['field_definitions' => $field_data]);

    } catch (\Exception $e) {
      return ExecutableResult::failure($this->t('Error retrieving field definitions: @message', [
        '@message' => $e->getMessage()
      ]));
    }
  }

  /**
   * Get field information including metadata and example values.
   */
  private function getFieldInfo(FieldDefinitionInterface $field_definition, string $entity_type_id, string $bundle): array {
    $field_type = $field_definition->getType();
    $settings = $field_definition->getSettings();

    $info = [
      'name' => $field_definition->getName(),
      'label' => $field_definition->getLabel(),
      'description' => $field_definition->getDescription(),
      'type' => $field_type,
      'required' => $field_definition->isRequired(),
      'cardinality' => $field_definition->getFieldStorageDefinition()->getCardinality(),
      'settings' => $settings,
    ];

    // Add translatable information if content translation is enabled.
    if ($this->moduleHandler->moduleExists('content_translation') && $this->contentTranslationManager) {
      $info['translatable'] = $this->isFieldTranslatable($field_definition, $entity_type_id, $bundle);
    }

    return $info;
  }

  /**
   * Check if a field is translatable.
   */
  private function isFieldTranslatable(FieldDefinitionInterface $field_definition, string $entity_type_id, string $bundle): bool {
    // Check if translation is enabled for this entity type and bundle.
    if (!$this->contentTranslationManager->isEnabled($entity_type_id, $bundle)) {
      return FALSE;
    }

    // Check if the field is translatable.
    return $field_definition->isTranslatable();
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    // todo Come back to.
    return $return_as_object ? AccessResult::allowed() : AccessResult::allowed()->isAllowed();
  }

}
