<?php

declare(strict_types=1);

namespace Drupal\tool_content\Plugin\tool\Tool;

use Drupal\content_translation\ContentTranslationManagerInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\field\FieldConfigInterface;
use Drupal\tool\Tool\ToolBase;
use Drupal\tool\TypedData\InputDefinition;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\TypedData\MapInputDefinition;
use Drupal\tool_entity\Plugin\tool\Tool\EntityBundleFieldDefinitions;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the field definitions by entity tool.
 */
#[Tool(
  id: 'entity_field_definitions',
  label: new TranslatableMarkup('Entity Field Definitions'),
  description: new TranslatableMarkup('Get base field and field definitions and value schema for a given entity and bundle. Usually used before creating or modifying entity content and field values.'),
  input_definitions: [
    'entity' => new InputDefinition(
      data_type: 'entity',
      label: new TranslatableMarkup("Entity"),
      description: new TranslatableMarkup("The entity to get field definitions for.")
    ),
  ],
  output_definitions: [
    'base_field_definitions' => new ContextDefinition(
      data_type: 'map',
      label: new TranslatableMarkup("Base Field Definitions"),
      description: new TranslatableMarkup("Array of entity base field definitions with value schema.")
    ),
    'field_definitions' => new ContextDefinition(
      data_type: 'map',
      label: new TranslatableMarkup("Field Definitions"),
      description: new TranslatableMarkup("Array of bundle specific field definitions with value schema.")
    ),
  ],
)]
class EntityFieldDefinitions extends ToolBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a new EntityBundleFieldDefinitions instance.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin ID.
   * @param array $plugin_definition
   *   The plugin definition.
   * @param EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager service.
   * @param EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param EntityTypeBundleInfoInterface $entityTypeBundleInfo
   *   The entity type bundle info service.
   * @param ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   * @param ContentTranslationManagerInterface|null $contentTranslationManager
   *   The content translation manager service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected EntityFieldManagerInterface $entityFieldManager,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EntityTypeBundleInfoInterface $entityTypeBundleInfo,
    protected ModuleHandlerInterface $moduleHandler,
    protected ?ContentTranslationManagerInterface $contentTranslationManager = NULL,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_field.manager'),
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('module_handler'),
      $container->has('content_translation.manager') ? $container->get('content_translation.manager') : NULL
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    ['entity' => $entity] = $values;
    $entity_type_id = $entity->getEntityTypeId();
    $bundle = $entity->bundle();

    try {
      // Validate entity type exists
      if (!$this->entityTypeManager->hasDefinition($entity_type_id)) {
        return ExecutableResult::failure($this->t('Entity type "@type" does not exist.', [
          '@type' => $entity_type_id
        ]));
      }

      // Validate bundle exists
      $bundle_info = $this->entityTypeBundleInfo->getBundleInfo($entity_type_id);
      if (!isset($bundle_info[$bundle])) {
        return ExecutableResult::failure($this->t('Bundle "@bundle" does not exist for entity type "@type".', [
          '@bundle' => $bundle,
          '@type' => $entity_type_id
        ]));
      }
      // Get base field definitions

      $base_field_definitions = $this->entityFieldManager->getBaseFieldDefinitions($entity_type_id);
      $base_field_info = [];
      $base_field_input_definition = EntityStub::getBaseFieldInputDefinition($entity_type_id);
      foreach ($base_field_definitions as $field_name => $field_definition) {
        $base_field_info[$field_name] = $this->getFieldInfo($field_definition, $entity_type_id, $bundle);
        if ($property_definition = $base_field_input_definition->getPropertyDefinition($field_name)) {
          $results = \Drupal::service('ai.context_definition_normalizer')->normalize([$property_definition]);
          $base_field_info[$field_name]['value_schema'] = $results[0]->renderPropertyArray();
        }
      }

      // Get field definitions
      $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle);
      $field_definitions = array_diff_key($field_definitions, $base_field_definitions);

      $field_info = [];
      foreach ($field_definitions as $field_name => $field_definition) {
        if (!$field_definition instanceof FieldConfigInterface) {
          continue; // Skip if not a valid field definition
        }
        $field_info[$field_name] = $this->getFieldInfo($field_definition, $entity_type_id, $bundle);
        $field_input_definition = FieldSetValue::getFieldInputDefinition($field_definition);
        $field_input_definition->setDefaultValue($field_definition->getDefaultValue($entity));
        $results = \Drupal::service('ai.context_definition_normalizer')->normalize([$field_input_definition]);
        $field_info[$field_name]['value_schema'] = $results[0]->renderPropertyArray();

      }

      return ExecutableResult::success($this->t('Found @base_count base field definitions and @count field definitions for @type:@bundle', [
        '@base_count' => count($base_field_info),
        '@count' => count($field_info),
        '@type' => $entity_type_id,
        '@bundle' => $bundle
      ]), [
        'base_field_definitions' => $base_field_info,
        'field_definitions' => $field_info
      ]);

    } catch (\Exception $e) {
      return ExecutableResult::failure($this->t('Error retrieving field definitions: @message', [
        '@message' => $e->getMessage()
      ]));
    }
  }

  /**
   * Get field information including metadata and example values.
   */
  private function getFieldInfo(FieldDefinitionInterface $field_definition, string $entity_type_id, string $bundle): array {
    $field_type = $field_definition->getType();
    $settings = $field_definition->getSettings();

    $info = [
      'name' => $field_definition->getName(),
      'label' => $field_definition->getLabel(),
      'description' => $field_definition->getDescription(),
      'type' => $field_type,
      'required' => $field_definition->isRequired(),
      'cardinality' => $field_definition->getFieldStorageDefinition()->getCardinality(),
      'settings' => $settings,
    ];

    // Add translatable information if content translation is enabled.
    if ($this->moduleHandler->moduleExists('content_translation') && $this->contentTranslationManager) {
      $info['translatable'] = $this->isFieldTranslatable($field_definition, $entity_type_id, $bundle);
    }
    return $info;
  }

  /**
   * Check if a field is translatable.
   */
  private function isFieldTranslatable(FieldDefinitionInterface $field_definition, string $entity_type_id, string $bundle): bool {
    // Check if translation is enabled for this entity type and bundle.
    if (!$this->contentTranslationManager->isEnabled($entity_type_id, $bundle)) {
      return FALSE;
    }

    // Check if the field is translatable.
    return $field_definition->isTranslatable();
  }

  /**
   * Get the field schema from the field item definition.
   */
  private function getFieldSchema(FieldDefinitionInterface $field_definition): array {
    try {

      $property_definitions = [];
      $typed_data_manager = \Drupal::service('typed_data_manager');

      // Create a detached item (not bound to an entity).
      $field_item = $typed_data_manager->create(
        $field_definition->getItemDefinition()
      );

      $item_properties = $field_item->getProperties();
      foreach ($field_definition->getItemDefinition()->getPropertyDefinitions() as $property_name => $property_definition) {
        if ($property_definition->isComputed() || !isset($item_properties[$property_name])) {
          continue; // Skip computed properties
        }
        $property_definitions[$property_name] = new InputDefinition(
          data_type: $property_definition->getDataType(),
          label: $property_definition->getLabel(),
          required: $property_definition->isRequired(),
          description: $property_definition->getDescription(),
          constraints: $property_definition->getConstraints()
        );
      }

      $input_definition =  new MapInputDefinition(
        data_type: 'map',
        label: $field_definition->getLabel(),
        required: $field_definition->isRequired(),
        multiple: $field_definition->getFieldStorageDefinition()->getCardinality() !== 1,
        description: new TranslatableMarkup('The value(s) to set for the field "@field".', ['@field' => $field_definition->getName()]),
        property_definitions: $property_definitions
      );
      $results = \Drupal::service('ai.context_definition_normalizer')->normalize([$input_definition]);
      $schema = $results[0]->renderPropertyArray();

      return $schema;
    } catch (\Exception $e) {
      return ['error' => 'Could not retrieve field schema: ' . $e->getMessage()];
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    // todo Come back to.
    return $return_as_object ? AccessResult::allowed() : AccessResult::allowed()->isAllowed();
  }

}
