<?php

declare(strict_types=1);

namespace Drupal\tool_content\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\TypedData\InputDefinition;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ToolBase;

use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the entity list tool.
 */
#[Tool(
  id: 'entity_list',
  label: new TranslatableMarkup('Entity List'),
  description: new TranslatableMarkup('List content entities with filtering, sorting, and field selection.'),
  input_definitions: [
    'entity_type_id' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Entity Type"),
      required: true,
      description: new TranslatableMarkup("The data name of the entity type you are trying to fetch entities for.")
    ),
    'bundle' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Bundle"),
      required: false,
      description: new TranslatableMarkup("The data name of the bundle type you want to get values for.")
    ),
    'entity_id' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Entity ID"),
      required: false,
      description: new TranslatableMarkup("The entity id to load a specific entity for.")
    ),
    'amount' => new InputDefinition(
      data_type: 'integer',
      label: new TranslatableMarkup("Amount"),
      required: false,
      description: new TranslatableMarkup("The amount of entities to fetch. 0 means all."),
      default_value: 10
    ),
    'offset' => new InputDefinition(
      data_type: 'integer',
      label: new TranslatableMarkup("Offset"),
      required: false,
      description: new TranslatableMarkup("The offset of entities to fetch."),
      default_value: 0
    ),
    'fields' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Fields"),
      required: false,
      multiple: true,
      description: new TranslatableMarkup("The fields to include in the output. Leave empty to include all fields."),
    ),
    'sort_field' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Sort Field"),
      required: false,
      description: new TranslatableMarkup("The field to sort by.")
    ),
    'sort_order' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Sort Order"),
      required: false,
      description: new TranslatableMarkup("The sort order (ASC or DESC)."),
      default_value: 'DESC'
    ),
  ],
  output_definitions: [
    'results' => new ContextDefinition(
      data_type: 'map',
      label: new TranslatableMarkup("Results"),
      description: new TranslatableMarkup("Array of entity data matching the criteria.")
    ),
  ],
)]
class EntityList extends ToolBase implements ContainerFactoryPluginInterface {

  /**
   * Constructor for the EntityList tool.
   *
   * @param array $configuration
   *  The configuration array.
   * @param $plugin_id
   *  The plugin ID.
   * @param $plugin_definition
   *  The plugin definition.
   * @param \Drupal\tool\TypedInputs $typed_inputs
   *  The typed inputs service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *  The entity type manager service.
   */
  public function __construct(
    array $configuration,
          $plugin_id,
          $plugin_definition,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    [
      'entity_type_id' => $entity_type_id,
      'bundle' => $bundle,
      'entity_id' => $entity_id,
      'amount' => $amount,
      'offset' => $offset,
      'fields' => $fields,
      'sort_field' => $sort_field,
      'sort_order' => $sort_order,
    ] = $values;

    try {
      // Validate entity type exists
      if (!$this->entityTypeManager->hasDefinition($entity_type_id)) {
        return ExecutableResult::failure($this->t('Entity type "@type" does not exist.', [
          '@type' => $entity_type_id
        ]));
      }

      $storage = $this->entityTypeManager->getStorage($entity_type_id);
      $entity_type_info = $this->entityTypeManager->getDefinition($entity_type_id);
      $bundle_key = $entity_type_info->getKey('bundle');

      // If in fields array 'label', replace with label for entity type.
      $key = array_search('label', $fields, true);
      if ($key !== false) {
        $fields[$key] =  $entity_type_info->getKey('label');
      }

      // If in fields array 'label', replace with label for entity type.
      $key = array_search('id', $fields, true);
      if ($key !== false) {
        $fields[$key] =  $entity_type_info->getKey('id');
      }

      // Build the query
      $query = $storage->getQuery();
      $query->accessCheck(TRUE);

      // Apply sorting
      if ($sort_field && $sort_order) {
        $query->sort($sort_field, $sort_order);
      }

      // Filter by bundle if specified
      if ($bundle && $bundle_key) {
        $query->condition($bundle_key, $bundle);
      }

      // Filter by entity ID if specified
      if ($entity_id) {
        $query->condition($entity_type_info->getKey('id'), (int) $entity_id);
      }

      // Apply range/limit
      if ($amount > 0) {
        $query->range($offset, $amount);
      }

      // Execute query
      $entity_ids = $query->execute();

      if (empty($entity_ids)) {
        return ExecutableResult::success($this->t('No entities found matching the criteria.'), [
          'entities' => []
        ]);
      }

      /** @var \Drupal\Core\Entity\ContentEntityInterface[] $entities */
      $entities = $storage->loadMultiple($entity_ids);
      $entity_list = [];
      $access_denied_count = 0;

      foreach ($entities as $entity) {
        // Double-check access (query access check may not be sufficient)
        if (!$entity->access('view')) {
          $access_denied_count++;
          continue;
        }

        // Ensure it's a content entity
        if (!$entity instanceof ContentEntityInterface) {
          continue;
        }

        $entity_data = [];

        if (!empty($fields)) {
          // Include only specified fields
          foreach ($fields as $field) {
            if ($entity->hasField($field)) {
              // Check field-level access
              if ($entity->get($field)->access('view')) {
                $entity_data[$field] = $this->getFieldValue($entity->get($field));
              }
            }
          }
        }
//        else {
//          // Include all fields with access checking
//          foreach ($entity->getFieldDefinitions() as $field_name => $field_definition) {
//            if ($entity->get($field_name)->access('view')) {
//              $entity_data[$field_name] = $this->getFieldValue($entity->get($field_name));
//            }
//          }
//        }

        // Add entity metadata
        $entity_data['_metadata'] = [
          'id' => $entity->id(),
          'type' => $entity->getEntityTypeId(),
          'bundle' => $entity->bundle(),
          'label' => $entity->label(),
          'uuid' => $entity->uuid(),
        ];

        $entity_list[] = $entity_data;
      }

      $total_count =  $this->entityTypeManager
        ->getStorage($entity_type_id)
        ->getQuery()
        ->count()
        ->accessCheck(FALSE)
        ->execute();

      $message = $this->t('Returned @count %entity_type_label(%entity_type_id) entities out of a total @total', [
          '@count' => count($entity_list),
          '%entity_type_label' => $entity_type_info->getLabel(),
          '%entity_type_id' => $entity_type_id,
          '@total' => $total_count
        ]);

      if ($access_denied_count > 0) {
        $message .= $this->t(' (@denied entities denied access)', [
          '@denied' => $access_denied_count
        ]);
      }

      return ExecutableResult::success($message, ['results' => $entity_list]);

    } catch (\Exception $e) {
      return ExecutableResult::failure($this->t('Error listing entities: @message', [
        '@message' => $e->getMessage()
      ]));
    }
  }

  /**
   * Extract field value in a consistent format.
   */
  private function getFieldValue($field_items) {
    if ($field_items->isEmpty()) {
      return null;
    }

    $field_definition = $field_items->getFieldDefinition();
    $cardinality = $field_definition->getFieldStorageDefinition()->getCardinality();

    if ($cardinality === 1) {
      return $this->getFieldItemValue($field_items->first());
    }

    $values = [];
    foreach ($field_items as $field_item) {
      $values[] = $this->getFieldItemValue($field_item);
    }
    return $values;
  }

  /**
   * Extract the appropriate value from a field item.
   */
  private function getFieldItemValue($field_item) {
    if (!$field_item) {
      return null;
    }

    $properties = $field_item->getProperties();

    // For simple fields, try to get the 'value' property first
    if (isset($properties['value'])) {
      return $field_item->value;
    }

    // For entity reference fields, get target_id and basic entity info
    if (isset($properties['target_id'])) {
      $data = ['target_id' => $field_item->target_id];
      if ($field_item->entity) {
        $data['entity'] = [
          'id' => $field_item->entity->id(),
          'label' => $field_item->entity->label(),
          'type' => $field_item->entity->getEntityTypeId(),
        ];
      }
      return $data;
    }

    // For complex fields, return all non-computed properties
    $values = [];
    foreach ($properties as $property_name => $property) {
      if (!$property->getDataDefinition()->isComputed()) {
        $values[$property_name] = $property->getValue();
      }
    }

    return $values ?: $field_item->getValue();
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    // Basic permission check - more specific access is done per entity
    $account = $account ?? \Drupal::currentUser();
    // Allow if user has general access to view content
    $access_result = AccessResult::allowedIfHasPermission($account, 'access content');
    return $return_as_object ? $access_result : $access_result->isAllowed();
  }

}
