<?php

namespace Drupal\top10_google_news\Form;

use Drupal;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 *
 */
class GoogleNewsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'top10_google_news.adminsettings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'google_news_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('top10_google_news.adminsettings');
    $news_categories = $this->getNewsCategories();
    $countries = $this->getCountries();
    $languages = $this->getLanguages();
    $form['google_api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Google API Key'),
      '#description' => $this->t('To use Google News, a Google News API Key must be created. '
        . 'Generate your API Key in <a target="_blank" href="https://newsapi.org/register">newsapi</a> on Google News. Enter your API Key here.'),
      '#default_value' => $config->get('google_api_key'),
      '#size' => 20,
      '#maxlength' => 50,
      '#required' => TRUE,
    ];
    $form['api_version'] = [
      '#type' => 'select',
      '#title' => $this->t('API Version'),
      '#description' => $this->t('To use Google News, a Google News API must be created. '
        . 'Select your Google News API version here. If your version is prior v1 select v2 here.'),
      '#default_value' => $config->get('api_version'),
      '#options' => ['v2' => 'v2'],
    ];
    $form['content_block'] = [
      '#type' => 'fieldset',
      '#title' => t('Content Block settings'),
      '#collapsible' => TRUE,
      '#collapsed' => FALSE,
      '#attributes' => [
        'class' => ['content__block'],
      ],
    ];
    $form['content_block']['content_news_category'] = [
      '#type' => 'select',
      '#title' => $this->t('News Category'),
      '#description' => $this->t('Select the Category for type of news you want to view. By default it will select "General" category.'),
      '#default_value' => !empty($config->get('content_news_category')) ? $config->get('content_news_category') : 'general',
      '#options' => $news_categories
    ];
    $form['content_block']['content_language'] = [
      '#type' => 'select',
      '#title' => $this->t('Language'),
      '#description' => $this->t('Select the type of language for news. By default it will select "English" language.'),
      '#default_value' => !empty($config->get('content_language')) ? $config->get('content_language') : 'en',
      '#options' => $languages
    ];
    $form['content_block']['content_country'] = [
      '#type' => 'select',
      '#title' => $this->t('Country'),
      '#description' => $this->t('Select the Country for which you want to view news. By default it will select "India" country.'),
      '#default_value' => !empty($config->get('content_country')) ? $config->get('content_country') : 'in',
      '#options' => $countries,
    ];
    $form['sidebar_block'] = [
      '#type' => 'fieldset',
      '#title' => t('Sidebar Block settings'),
      '#collapsible' => TRUE,
      '#collapsed' => FALSE,
      '#attributes' => [
        'class' => ['sidebar__block'],
      ],
    ];
    $form['sidebar_block']['sidebar_news_category'] = [
      '#type' => 'select',
      '#title' => $this->t('News Category'),
      '#description' => $this->t('Select the Category for type of news you want to view. By default it will select "General" category.'),
      '#default_value' => !empty($config->get('sidebar_news_category')) ? $config->get('sidebar_news_category') : 'general',
      '#options' => $news_categories
    ];
    $form['sidebar_block']['sidebar_language'] = [
      '#type' => 'select',
      '#title' => $this->t('Language'),
      '#description' => $this->t('Select the type of language for news. By default it will select "English" language.'),
      '#default_value' => !empty($config->get('sidebar_language')) ? $config->get('sidebar_language') : 'en',
      '#options' => $languages
    ];
    $form['sidebar_block']['sidebar_country'] = [
      '#type' => 'select',
      '#title' => $this->t('Country'),
      '#description' => $this->t('Select the Country for which you want to view news. '),
      '#default_value' => !empty($config->get('sidebar_country')) ? $config->get('sidebar_country') : 'in',
      '#options' => $countries,
    ];
    $form['footer_block'] = [
      '#type' => 'fieldset',
      '#title' => t('Footer Block settings'),
      '#collapsible' => TRUE,
      '#collapsed' => FALSE,
      '#attributes' => [
        'class' => ['footer__block'],
      ],
    ];
    $form['footer_block']['footer_news_category'] = [
      '#type' => 'select',
      '#title' => $this->t('News Category'),
      '#description' => $this->t('Select the Category for type of news you want to view. By default it will select "General" category.'),
      '#default_value' => !empty($config->get('footer_news_category')) ? $config->get('footer_news_category') : 'general',
      '#options' => $news_categories
    ];
    $form['footer_block']['footer_language'] = [
      '#type' => 'select',
      '#title' => $this->t('Language'),
      '#description' => $this->t('Select the type of language for news. By default it will select "English" language.'),
      '#default_value' => !empty($config->get('footer_language')) ? $config->get('footer_language') : 'en',
      '#options' => $languages
    ];
    $form['footer_block']['footer_country'] = [
      '#type' => 'select',
      '#title' => $this->t('Country'),
      '#description' => $this->t('Select the Country for which you want to view news. By default it will select "India" country.'),
      '#default_value' => !empty($config->get('footer_country')) ? $config->get('footer_country') : 'in',
      '#options' => $countries,
    ];
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    if ($form_state->getValue('content_news_category') && $form_state->getValue('content_language')) {
      $str = "https://newsapi.org/" . $form_state->getValue('api_version') .
        "/top-headlines?country=" . $form_state->getValue('content_country') .
        "&category=" . $form_state->getValue('content_news_category') .
        "&language=" . $form_state->getValue('content_language') . "&apiKey=" .
        $form_state->getValue('google_api_key');
    }
    if (strlen($form_state->getValue('google_api_key')) != 32) {
      $form_state->setErrorByName('google_api_key', $this->t('The Google News API Key does not appear to be valid. It is usually a 32 character hash.'));
    }
    try {
      $response = \Drupal::httpClient()->get($str)->getBody()->getContents();
    }
    catch (\Throwable $th) {
      $form_state->setErrorByName('api_version', $this->t('The selected Version is not working.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);

    $this->config('top10_google_news.adminsettings')
      ->set('google_api_key', $form_state->getValue('google_api_key'))
      ->set('api_version', $form_state->getValue('api_version'))
      ->set('content_news_category', $form_state->getValue('content_news_category'))
      ->set('content_language', $form_state->getValue('content_language'))
      ->set('content_country', $form_state->getValue('content_country'))
      ->set('footer_news_category', $form_state->getValue('footer_news_category'))
      ->set('footer_language', $form_state->getValue('footer_language'))
      ->set('footer_country', $form_state->getValue('footer_country'))
      ->set('sidebar_news_category', $form_state->getValue('sidebar_news_category'))
      ->set('sidebar_language', $form_state->getValue('sidebar_language'))
      ->set('sidebar_country', $form_state->getValue('sidebar_country'))
      ->save();
  }

  /**
   * Get the list of countries.
   * @return $countries
   */
  public function getCountries() {
    $countries = [
      '' => 'Default',
      'ae' => 'United Arab Emirates',
      'ar' => '	Argentina',
      'at' => 'Austria',
      'be' => 'Belgium',
      'au' => 'Australia',
      'br' => 'Brazil',
      'bg' => 'Bulgaria',
      'ca' => 'Canada',
      'cn' => 'China',
      'co' => 'Colombia',
      'cu' => 'Cuba',
      'cz' => 'Czech Republic',
      'eg' => 'Egypt',
      'fr' => 'France',
      'de' => 'Germany',
      'gr' => 'Greece',
      'hk' => 'Hong Kong',
      'hu' => 'Hungary',
      'in' => 'India',
      'id' => 'Indonesia',
      'ie' => 'Ireland',
      'il' => 'Israel',
      'it' => 'Italy',
      'JM' => 'Jamaica',
      'kr' => 'Korea',
      'lv' => 'Latvia',
      'lt' => 'Lithuania',
      'my' => 'Malaysia',
      'mx' => 'Mexico',
      'ma' => 'Mozambique',
      'nl' => 'Netherlands',
      'nz' => 'New Zealand',
      'ng' => 'Nigeria',
      'no' => 'Norway',
      'ph' => 'Philippines',
      'PN' => 'Pitcairn',
      'pl' => 'Poland',
      'pt' => 'Portugal',
      'ro' => 'Romania',
      'ru' => 'Russian Federation',
      'sa' => 'Saudi Arabia',
      'rs' => 'Serbia',
      'sg' => 'Singapore',
      'sk' => 'Slovakia',
      'si' => 'Slovenia',
      'za' => 'South Africa',
      'se' => 'Sweden',
      'ch' => 'Switzerland',
      'tw' => 'Taiwan',
      'th' => 'Thailand',
      'tr' => 'Turkey',
      'ua' => 'Ukraine',
      'gb' => 'United Kingdom',
      'us' => 'United States',
      've' => 'Venezuela',
    ];
    return $countries;
  }

  /**
   * Get the list of News Categories.
   * @return $news_categories
   */
  public function getNewsCategories() {
    $news_categories = [
      '' => 'Default',
      'business' => 'Business',
      'entertainment' => 'Entertainment',
      'general' => 'General',
      'health' => 'Health',
      'science' => 'Science',
      'sports' => 'Sports',
      'technology' => 'Technology'
    ];
    return $news_categories;
  }

  /**
   * Get list of languages.
   * @return $languages
   */
  public function getLanguages() {
    $languages = [
      '' => 'Default',
      'ar' => 'Argentina',
      'de' => 'Germany',
      'en' => 'English',
      'fr' => 'French',
      'it' => 'Italy',
      'nl' => 'Netherlands',
      'no' => 'Norway',
      'pt' => 'Portugal',
      'ru' => 'Russian'
    ];
    return $languages;
  }

}
