<?php

namespace Drupal\tour_extras_url_step\Plugin\tour\tip;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Utility\Token;
use Drupal\tour\Attribute\Tip;
use Drupal\tour\TipPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Displays a URL link as a tip.
 *
 * WARNING: This plugin should always be used as the last step in a tour.
 * It replaces the "next" button with a custom URL link.
 */
#[Tip(
  id: 'url_step',
  title: new TranslatableMarkup('URL Step'),
)]
class UrlStep extends TipPluginBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a UrlStep object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Utility\Token $token
   *   The token service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, protected Token $token) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static($configuration, $plugin_id, $plugin_definition, $container->get('token'));
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'url' => '',
      'next_button_label' => 'Next',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function getBody(): array {
    // This URL step doesn't render visible content.
    // The JavaScript library and the Alter hook handle the URL redirection:
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    \Drupal::messenger()->addWarning('This step should be configured as the last step in the tour!');
    $id = $this->get('id');
    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Title'),
      '#description' => $this->t('The tip title should match the title of the component the selector field is referencing. For the initial general tip with the empty selector field, use the h1 of the page. Each additional tip without a selector should have a unique descriptive title.'),
      '#required' => TRUE,
      '#default_value' => $this->get('label'),
    ];
    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $id,
      '#disabled' => !empty($id),
      '#machine_name' => [
        'exists' => '\Drupal\tour\Entity\Tour::load',
      ],
    ];
    $form['plugin'] = [
      '#type' => 'value',
      '#value' => $this->get('plugin'),
    ];
    $form['weight'] = [
      '#type' => 'weight',
      '#title' => $this->t('Weight'),
      '#default_value' => $this->get('weight'),
      '#attributes' => [
        'class' => ['tip-order-weight'],
      ],
    ];

    $form['url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('URL'),
      '#required' => TRUE,
      '#default_value' => $this->get('url'),
      '#description' => $this->t('The relative URL to link to (e.g., /admin/content, /node/123). You can use the "?tips" and "?tours" query parameters to redirect to a specific tip or tour. Supports tokens.'),
    ];

    $form['next_button_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Next Button Label'),
      '#required' => TRUE,
      '#default_value' => $this->get('next_button_label'),
      '#description' => $this->t('The text to display for the next button. Defaults to "Next".'),
    ];

    // Add token tree if the token module is available.
    if (\Drupal::moduleHandler()->moduleExists('token')) {
      $form['token_tree'] = [
        '#theme' => 'token_tree_link',
        '#token_types' => ['current-user', 'site', 'current-page'],
        '#global_types' => TRUE,
        '#show_restricted' => TRUE,
        '#weight' => 90,
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->configuration['url'] = $form_state->getValue('url');
    $this->configuration['next_button_label'] = $form_state->getValue('next_button_label');
  }

}
