<?php

namespace Drupal\tour_extras_wysiwyg\Plugin\tour\tip;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tour\Attribute\Tip;
use Drupal\tour\Plugin\tour\tip\TipPluginText;

/**
 * Displays formatted text as a tip with WYSIWYG support.
 */
#[Tip(
  id: 'text_formatted',
  title: new TranslatableMarkup('Formatted Text'),
)]
class TipPluginTextFormatted extends TipPluginText implements ContainerFactoryPluginInterface {

  /**
   * The text format for the body.
   *
   * @var string
   */
  protected string $format;

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    $parentDefaultConfiguration = parent::defaultConfiguration();
    $parentDefaultConfiguration['body'] = ['value' => '', 'format' => filter_default_format()];
    return $parentDefaultConfiguration;
  }

  /**
   * {@inheritdoc}
   */
  public function getBody(): array {
    $text = $this->token->replace($this->get('body')['value']);
    $format = $this->get('body')['format'];

    return [
      '#type' => 'processed_text',
      '#text' => $text,
      '#format' => $format,
      '#attributes' => [
        'class' => [
          'tour-tip-body',
          'tour-tip-body--formatted-text',
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    // Use text_format field instead of textarea for WYSIWYG support.
    $form['body'] = [
      '#type' => 'text_format',
      '#title' => $this->t('Body'),
      '#required' => TRUE,
      '#default_value' => $this->get('body')['value'],
      '#format' => $this->get('body')['format'],
      '#description' => $this->t('The tip body content.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->configuration['body'] = $form_state->getValue('body');
  }

}
