<?php

namespace Drupal\tour_extras_url_step\Plugin\tour\tip;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Utility\Token;
use Drupal\tour\Attribute\Tip;
use Drupal\tour\TipPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Displays a URL link as a tip.
 *
 * WARNING: This plugin should always be used as the last step in a tour.
 * It replaces the "next" button with a custom URL link.
 */
#[Tip(
  id: 'url_step',
  title: new TranslatableMarkup('URL Step'),
)]
class UrlStep extends TipPluginBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a UrlStep object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Utility\Token $token
   *   The token service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, protected Token $token) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static($configuration, $plugin_id, $plugin_definition, $container->get('token'));
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    $config = [
      'url' => '',
      'next_button_label' => 'Next',
    ] + parent::defaultConfiguration();
    // Unset unnecessary configs:
    unset($config['selector']);
    unset($config['position']);
    return $config;
  }

  /**
   * {@inheritdoc}
   */
  public function getBody(): array {
    // This URL step doesn't render visible content.
    // The JavaScript library and the Alter hook handle the URL redirection:
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    \Drupal::messenger()->addWarning('This step should be configured as the last step in the tour!');
    $form = parent::buildConfigurationForm($form, $form_state);
    // Unset unnecessary configs:
    unset($form['selector']);
    unset($form['position']);

    // @improve: Use the same widget as menu items use for links:
    $form['url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('URL'),
      '#required' => TRUE,
      '#default_value' => $this->get('url'),
      '#description' => $this->t('The relative URL to link to (e.g., /admin/content, /node/123). You can use the "?tips" and "?tours" query parameters to redirect to a specific tip or tour.'),
    ];

    $form['next_button_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Next Button Label'),
      '#required' => TRUE,
      '#default_value' => $this->get('next_button_label'),
      '#description' => $this->t('The text to display for the next button. Defaults to "Next". Supports tokens.'),
    ];

    // Add token tree if the token module is available.
    if (\Drupal::moduleHandler()->moduleExists('token')) {
      $form['token_tree'] = [
        '#theme' => 'token_tree_link',
        '#token_types' => ['current-user', 'site', 'current-page'],
        '#global_types' => TRUE,
        '#show_restricted' => TRUE,
        '#weight' => 90,
      ];
    }
    return $form;
  }

}
