/**
 * @file
 * AJAX functionality for mail log list page.
 */

(function (Drupal, drupalSettings, once) {
  'use strict';

  /**
   * Attach AJAX behaviors to mail log list.
   */
  Drupal.behaviors.mailLogAjax = {
    attach: function (context) {
      // Only run once on the main container.
      const container = once('mail-log-ajax', '#mail-log-results', context);
      if (!container.length) {
        return;
      }

      const wrapper = container[0];

      // Intercept filter form submission.
      this.attachFilterForm(wrapper);

      // Intercept sort link clicks.
      this.attachSortLinks(wrapper);

      // Intercept pagination clicks.
      this.attachPagerLinks(wrapper);

      // Intercept items per page change.
      this.attachLimitSelect(wrapper);
    },

    /**
     * Attach AJAX to filter form.
     */
    attachFilterForm: function (wrapper) {
      const form = wrapper.querySelector('.mail-log-filters-form');
      if (!form) {
        return;
      }

      form.addEventListener('submit', function (e) {
        e.preventDefault();
        const formData = new FormData(form);
        const params = new URLSearchParams(formData);

        // Preserve current sort and limit.
        const currentUrl = new URL(window.location.href);
        if (currentUrl.searchParams.has('sort')) {
          params.set('sort', currentUrl.searchParams.get('sort'));
        }
        if (currentUrl.searchParams.has('order')) {
          params.set('order', currentUrl.searchParams.get('order'));
        }
        if (currentUrl.searchParams.has('limit')) {
          params.set('limit', currentUrl.searchParams.get('limit'));
        }

        Drupal.behaviors.mailLogAjax.loadContent(params.toString());
      });
    },

    /**
     * Attach AJAX to sort links.
     */
    attachSortLinks: function (wrapper) {
      const sortLinks = wrapper.querySelectorAll('.mail-log-sort-link');

      sortLinks.forEach(function (link) {
        link.addEventListener('click', function (e) {
          e.preventDefault();
          const url = new URL(link.href);
          Drupal.behaviors.mailLogAjax.loadContent(url.search.substring(1));
        });
      });
    },

    /**
     * Attach AJAX to pager links.
     */
    attachPagerLinks: function (wrapper) {
      const pagerLinks = wrapper.querySelectorAll('.pager a');

      pagerLinks.forEach(function (link) {
        link.addEventListener('click', function (e) {
          e.preventDefault();
          const url = new URL(link.href);
          Drupal.behaviors.mailLogAjax.loadContent(url.search.substring(1));
        });
      });
    },

    /**
     * Attach AJAX to items per page select.
     */
    attachLimitSelect: function (wrapper) {
      const select = wrapper.querySelector('.mail-log-limit-select');
      if (!select) {
        return;
      }

      select.addEventListener('change', function () {
        const currentUrl = new URL(window.location.href);
        currentUrl.searchParams.set('limit', select.value);
        // Reset to first page when changing limit.
        currentUrl.searchParams.delete('page');
        Drupal.behaviors.mailLogAjax.loadContent(currentUrl.search.substring(1));
      });
    },

    /**
     * Load content via AJAX and update URL.
     */
    loadContent: function (queryString) {
      const wrapper = document.getElementById('mail-log-results');
      if (!wrapper) {
        return;
      }

      // Show loading state.
      wrapper.classList.add('mail-log-loading');

      // Build AJAX URL.
      const baseUrl = drupalSettings.path.baseUrl + drupalSettings.path.currentPath;
      const ajaxUrl = baseUrl + '?' + queryString + '&_wrapper_format=drupal_ajax';

      // Update browser URL.
      const newUrl = baseUrl + (queryString ? '?' + queryString : '');
      window.history.pushState({ path: newUrl }, '', newUrl);

      // Make AJAX request.
      fetch(ajaxUrl, {
        method: 'GET',
        headers: {
          'Accept': 'application/json'
        }
      })
      .then(function (response) {
        return response.json();
      })
      .then(function (data) {
        // Process AJAX commands.
        if (data && data.length > 0) {
          data.forEach(function (command) {
            if (command.command === 'insert' && command.method === 'replaceWith') {
              wrapper.outerHTML = command.data;
              // Re-attach behaviors.
              Drupal.attachBehaviors(document.getElementById('mail-log-results'));
            }
          });
        }
      })
      .catch(function (error) {
        console.error('Mail log AJAX error:', error);
        // Fallback to regular navigation.
        window.location.href = newUrl;
      })
      .finally(function () {
        const newWrapper = document.getElementById('mail-log-results');
        if (newWrapper) {
          newWrapper.classList.remove('mail-log-loading');
        }
      });
    }
  };

  /**
   * Handle browser back/forward navigation.
   */
  window.addEventListener('popstate', function (e) {
    if (e.state && e.state.path) {
      // Reload page on back/forward.
      window.location.reload();
    }
  });

})(Drupal, drupalSettings, once);
