<?php

declare(strict_types=1);

namespace Drupal\trace_mail_log\Controller;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Url;
use Drupal\trace_mail_log\Service\PurgeService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for the mail log dashboard.
 */
class DashboardController extends ControllerBase {

  /**
   * The database connection.
   */
  protected Connection $database;

  /**
   * The purge service.
   */
  protected PurgeService $purgeService;

  /**
   * The date formatter service.
   */
  protected DateFormatterInterface $dateFormatter;

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Constructs a DashboardController.
   */
  public function __construct(
    Connection $database,
    PurgeService $purge_service,
    DateFormatterInterface $date_formatter,
    ConfigFactoryInterface $config_factory,
  ) {
    $this->database = $database;
    $this->purgeService = $purge_service;
    $this->dateFormatter = $date_formatter;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('database'),
      $container->get('trace_mail_log.purge_service'),
      $container->get('date.formatter'),
      $container->get('config.factory'),
    );
  }

  /**
   * Displays the dashboard.
   */
  public function dashboard(): array {
    $stats = $this->purgeService->getStats();

    // Get recent failures.
    $recent_failures = $this->getRecentFailures(5);

    return [
      '#theme' => 'mail_log_dashboard',
      '#stats' => $stats,
      '#recent_failures' => $recent_failures,
      '#list_url' => Url::fromRoute('trace_mail_log.list')->toString(),
      '#attached' => [
        'library' => ['trace_mail_log/dashboard'],
      ],
    ];
  }

  /**
   * Gets recent failures.
   */
  protected function getRecentFailures(int $limit): array {
    $config = $this->configFactory->get('trace_mail_log.settings');
    $date_format = $config->get('date_format_list') ?: 'short';

    $query = $this->database->select('trace_mail_log', 'm')
      ->fields('m', ['id', 'recipients', 'subject', 'response_message', 'created'])
      ->condition('event_type', 'failed')
      ->orderBy('created', 'DESC')
      ->range(0, $limit);

    $results = $query->execute()->fetchAll();

    $failures = [];
    foreach ($results as $row) {
      $recipients = json_decode($row->recipients, TRUE);
      $to = $recipients['to'][0] ?? 'Unknown';

      $failures[] = [
        'id' => $row->id,
        'recipient' => $to,
        'subject' => $row->subject ?? 'No subject',
        'error' => $row->response_message ?? 'Unknown error',
        'date' => $this->dateFormatter->format((int) $row->created, $date_format),
        'url' => Url::fromRoute('trace_mail_log.detail', ['id' => $row->id])->toString(),
      ];
    }

    return $failures;
  }

}
