<?php

declare(strict_types=1);

namespace Drupal\trace_mail_log\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\symfony_mailer\EmailInterface;
use Drupal\trace_mail_log\Service\MailLogService;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\Mailer\Event\FailedMessageEvent;
use Symfony\Component\Mailer\Event\MessageEvent;
use Symfony\Component\Mailer\Event\SentMessageEvent;
use Symfony\Component\Mime\Email;

/**
 * Subscribes to Symfony Mailer events for logging.
 */
class MailerEventSubscriber implements EventSubscriberInterface {

  /**
   * Header name for storing our tracking UUID.
   */
  public const UUID_HEADER = 'X-NSI-Mail-Log-UUID';

  /**
   * Header name for storing the mail key.
   */
  public const MAIL_KEY_HEADER = 'X-Drupal-Mail-Key';

  /**
   * The mail log service.
   */
  protected MailLogService $mailLogService;

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Constructs a MailerEventSubscriber.
   */
  public function __construct(
    MailLogService $mail_log_service,
    ConfigFactoryInterface $config_factory,
  ) {
    $this->mailLogService = $mail_log_service;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      MessageEvent::class => ['onMessage', 0],
      SentMessageEvent::class => ['onSentMessage', 0],
      FailedMessageEvent::class => ['onFailedMessage', 0],
    ];
  }

  /**
   * Handles MessageEvent (before sending).
   *
   * Injects tracking UUID and logs "sending" event.
   */
  public function onMessage(MessageEvent $event): void {
    if (!$this->mailLogService->isEnabled()) {
      return;
    }

    $message = $event->getMessage();
    if (!$message instanceof Email) {
      return;
    }

    // Generate or retrieve UUID.
    $uuid = $this->getOrCreateUuid($message);

    // Extract data.
    $transport_dsn = $event->getTransport();
    $data = $this->mailLogService->extractEmailData($message, $transport_dsn);
    $data['mail_key'] = $this->getMailKey($message);

    // Log "sending" event.
    $this->mailLogService->log($uuid, 'sending', 'pending', $data);
  }

  /**
   * Handles SentMessageEvent (after successful send).
   */
  public function onSentMessage(SentMessageEvent $event): void {
    if (!$this->mailLogService->isEnabled()) {
      return;
    }

    $sentMessage = $event->getMessage();
    $originalMessage = $sentMessage->getOriginalMessage();

    if (!$originalMessage instanceof Email) {
      return;
    }

    // Get UUID.
    $uuid = $this->getUuid($originalMessage);
    if (!$uuid) {
      return;
    }

    // Extract data.
    $data = $this->mailLogService->extractEmailData($originalMessage);
    $data['mail_key'] = $this->getMailKey($originalMessage);
    $data['message_id'] = $sentMessage->getMessageId();

    // Get transcript from debug info.
    $debug = $sentMessage->getDebug();
    if ($debug) {
      $data['transcript'] = $debug;

      // Parse response from transcript.
      $response = $this->mailLogService->parseSmtpResponse($debug);
      $data['response_code'] = $response['code'];
      $data['response_message'] = $response['message'];
    }

    // Log "sent" event.
    $this->mailLogService->log($uuid, 'sent', 'success', $data);
  }

  /**
   * Handles FailedMessageEvent.
   */
  public function onFailedMessage(FailedMessageEvent $event): void {
    if (!$this->mailLogService->isEnabled()) {
      return;
    }

    $message = $event->getMessage();
    if (!$message instanceof Email) {
      return;
    }

    // Get UUID.
    $uuid = $this->getUuid($message);
    if (!$uuid) {
      return;
    }

    // Extract data.
    $data = $this->mailLogService->extractEmailData($message);
    $data['mail_key'] = $this->getMailKey($message);

    // Get error info.
    $error = $event->getError();
    $data['error'] = $error->getMessage();

    // Try to get transcript from exception if available.
    if (method_exists($error, 'getDebug')) {
      $data['transcript'] = $error->getDebug();
    }

    // Log "failed" event.
    $this->mailLogService->log($uuid, 'failed', 'failed', $data);
  }

  /**
   * Gets or creates a tracking UUID for the message.
   */
  protected function getOrCreateUuid(Email $message): string {
    $headers = $message->getHeaders();

    if ($headers->has(self::UUID_HEADER)) {
      return $headers->get(self::UUID_HEADER)->getBodyAsString();
    }

    $uuid = $this->mailLogService->generateUuid();
    $headers->addTextHeader(self::UUID_HEADER, $uuid);

    return $uuid;
  }

  /**
   * Gets the tracking UUID from a message.
   */
  protected function getUuid(Email $message): ?string {
    $headers = $message->getHeaders();

    if ($headers->has(self::UUID_HEADER)) {
      return $headers->get(self::UUID_HEADER)->getBodyAsString();
    }

    return NULL;
  }

  /**
   * Gets the Drupal mail key from message headers.
   */
  protected function getMailKey(Email $message): ?string {
    $headers = $message->getHeaders();

    if ($headers->has(self::MAIL_KEY_HEADER)) {
      return $headers->get(self::MAIL_KEY_HEADER)->getBodyAsString();
    }

    // Try to get from X-Mailer-Type which symfony_mailer sets.
    if ($headers->has('X-Mailer-Type')) {
      return $headers->get('X-Mailer-Type')->getBodyAsString();
    }

    return NULL;
  }

}
