<?php

declare(strict_types=1);

namespace Drupal\trace_mail_log\Form;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Settings form for mail log configuration.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The date formatter service.
   */
  protected DateFormatterInterface $dateFormatter;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = parent::create($container);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->dateFormatter = $container->get('date.formatter');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'trace_mail_log_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['trace_mail_log.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('trace_mail_log.settings');

    $form['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable mail logging'),
      '#description' => $this->t('When enabled, all email operations will be logged.'),
      '#default_value' => $config->get('enabled'),
    ];

    $form['retention_days'] = [
      '#type' => 'number',
      '#title' => $this->t('Retention period (days)'),
      '#description' => $this->t('Log entries older than this will be automatically purged. Set to 0 to disable automatic purging.'),
      '#default_value' => $config->get('retention_days'),
      '#min' => 0,
      '#max' => 365,
    ];

    $form['log_directory'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Log file directory'),
      '#description' => $this->t('Directory where transcript files are stored. Use private:// for security.'),
      '#default_value' => $config->get('log_directory'),
    ];

    $form['log_body'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Log email body'),
      '#description' => $this->t('Include email HTML and text body in log files. Disable for privacy.'),
      '#default_value' => $config->get('log_body'),
    ];

    $form['log_attachments'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Log attachments'),
      '#description' => $this->t('Include attachment information in logs. Note: Attachment content is not stored.'),
      '#default_value' => $config->get('log_attachments'),
    ];

    // Display settings.
    $form['display'] = [
      '#type' => 'details',
      '#title' => $this->t('Display settings'),
      '#open' => TRUE,
    ];

    // Build date format options from available Drupal date formats.
    $date_formats = $this->entityTypeManager->getStorage('date_format')->loadMultiple();
    $date_format_options = [];
    $example_timestamp = time();
    foreach ($date_formats as $id => $format) {
      $example = $this->dateFormatter->format($example_timestamp, $id);
      $date_format_options[$id] = $format->label() . ' (' . $example . ')';
    }

    $form['display']['date_format_list'] = [
      '#type' => 'select',
      '#title' => $this->t('Date format for list view'),
      '#description' => $this->t('Select the date format to use in the mail log list.'),
      '#options' => $date_format_options,
      '#default_value' => $config->get('date_format_list') ?: 'short',
    ];

    $form['display']['date_format_detail'] = [
      '#type' => 'select',
      '#title' => $this->t('Date format for detail view'),
      '#description' => $this->t('Select the date format to use in the mail log detail page.'),
      '#options' => $date_format_options,
      '#default_value' => $config->get('date_format_detail') ?: 'medium',
    ];

    $form['display']['items_per_page'] = [
      '#type' => 'number',
      '#title' => $this->t('Default items per page'),
      '#description' => $this->t('Default number of log entries to display per page. Users can override this temporarily.'),
      '#default_value' => $config->get('items_per_page') ?: 50,
      '#min' => 10,
      '#max' => 500,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('trace_mail_log.settings')
      ->set('enabled', (bool) $form_state->getValue('enabled'))
      ->set('retention_days', (int) $form_state->getValue('retention_days'))
      ->set('log_directory', $form_state->getValue('log_directory'))
      ->set('log_body', (bool) $form_state->getValue('log_body'))
      ->set('log_attachments', (bool) $form_state->getValue('log_attachments'))
      ->set('date_format_list', $form_state->getValue('date_format_list'))
      ->set('date_format_detail', $form_state->getValue('date_format_detail'))
      ->set('items_per_page', (int) $form_state->getValue('items_per_page'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
